/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.spark.network.buffer.FileSegmentManagedBuffer;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.shuffle.ExecutorDiskUtils;
import org.apache.spark.network.shuffle.ShuffleIndexInformation;
import org.apache.spark.network.shuffle.ShuffleIndexRecord;
import org.apache.spark.network.shuffle.protocol.ExecutorShuffleInfo;
import org.apache.spark.network.util.JavaUtils;
import org.apache.spark.network.util.LevelDBProvider;
import org.apache.spark.network.util.NettyUtils;
import org.apache.spark.network.util.TransportConf;
import org.iq80.leveldb.DB;
import org.iq80.leveldb.DBIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sparkproject.guava.annotations.VisibleForTesting;
import org.sparkproject.guava.cache.CacheBuilder;
import org.sparkproject.guava.cache.CacheLoader;
import org.sparkproject.guava.cache.LoadingCache;
import org.sparkproject.guava.cache.Weigher;
import org.sparkproject.guava.collect.Maps;

public class ExternalShuffleBlockResolver {
    private static final Logger logger = LoggerFactory.getLogger(ExternalShuffleBlockResolver.class);
    private static final ObjectMapper mapper = new ObjectMapper();
    private static final String APP_KEY_PREFIX = "AppExecShuffleInfo";
    private static final LevelDBProvider.StoreVersion CURRENT_VERSION = new LevelDBProvider.StoreVersion(1, 0);
    private static final Pattern MULTIPLE_SEPARATORS = Pattern.compile(File.separator + "{2,}");
    @VisibleForTesting
    final ConcurrentMap<AppExecId, ExecutorShuffleInfo> executors;
    private final LoadingCache<File, ShuffleIndexInformation> shuffleIndexCache;
    private final Executor directoryCleaner;
    private final TransportConf conf;
    private final boolean rddFetchEnabled;
    @VisibleForTesting
    final File registeredExecutorFile;
    @VisibleForTesting
    final DB db;
    private final List<String> knownManagers = Arrays.asList("org.apache.spark.shuffle.sort.SortShuffleManager", "org.apache.spark.shuffle.unsafe.UnsafeShuffleManager");

    public ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile) throws IOException {
        this(conf, registeredExecutorFile, Executors.newSingleThreadExecutor(NettyUtils.createThreadFactory((String)"spark-shuffle-directory-cleaner")));
    }

    @VisibleForTesting
    ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile, Executor directoryCleaner) throws IOException {
        this.conf = conf;
        this.rddFetchEnabled = Boolean.valueOf(conf.get("spark.shuffle.service.fetch.rdd.enabled", "false"));
        this.registeredExecutorFile = registeredExecutorFile;
        String indexCacheSize = conf.get("spark.shuffle.service.index.cache.size", "100m");
        CacheLoader<File, ShuffleIndexInformation> indexCacheLoader = new CacheLoader<File, ShuffleIndexInformation>(){

            public ShuffleIndexInformation load(File file) throws IOException {
                return new ShuffleIndexInformation(file);
            }
        };
        this.shuffleIndexCache = CacheBuilder.newBuilder().maximumWeight(JavaUtils.byteStringAsBytes((String)indexCacheSize)).weigher((Weigher)new Weigher<File, ShuffleIndexInformation>(){

            public int weigh(File file, ShuffleIndexInformation indexInfo) {
                return indexInfo.getSize();
            }
        }).build((CacheLoader)indexCacheLoader);
        this.db = LevelDBProvider.initLevelDB((File)this.registeredExecutorFile, (LevelDBProvider.StoreVersion)CURRENT_VERSION, (ObjectMapper)mapper);
        this.executors = this.db != null ? ExternalShuffleBlockResolver.reloadRegisteredExecutors(this.db) : Maps.newConcurrentMap();
        this.directoryCleaner = directoryCleaner;
    }

    public int getRegisteredExecutorsSize() {
        return this.executors.size();
    }

    public void registerExecutor(String appId, String execId, ExecutorShuffleInfo executorInfo) {
        AppExecId fullId = new AppExecId(appId, execId);
        logger.info("Registered executor {} with {}", (Object)fullId, (Object)executorInfo);
        if (!this.knownManagers.contains(executorInfo.shuffleManager)) {
            throw new UnsupportedOperationException("Unsupported shuffle manager of executor: " + executorInfo);
        }
        try {
            if (this.db != null) {
                byte[] key = ExternalShuffleBlockResolver.dbAppExecKey(fullId);
                byte[] value = mapper.writeValueAsString((Object)executorInfo).getBytes(StandardCharsets.UTF_8);
                this.db.put(key, value);
            }
        }
        catch (Exception e) {
            logger.error("Error saving registered executors", (Throwable)e);
        }
        this.executors.put(fullId, executorInfo);
    }

    public ManagedBuffer getBlockData(String appId, String execId, int shuffleId, long mapId, int reduceId) {
        return this.getContinuousBlocksData(appId, execId, shuffleId, mapId, reduceId, reduceId + 1);
    }

    public ManagedBuffer getContinuousBlocksData(String appId, String execId, int shuffleId, long mapId, int startReduceId, int endReduceId) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        return this.getSortBasedShuffleBlockData(executor, shuffleId, mapId, startReduceId, endReduceId);
    }

    public ManagedBuffer getRddBlockData(String appId, String execId, int rddId, int splitIndex) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        return this.getDiskPersistedRddBlockData(executor, rddId, splitIndex);
    }

    public void applicationRemoved(String appId, boolean cleanupLocalDirs) {
        logger.info("Application {} removed, cleanupLocalDirs = {}", (Object)appId, (Object)cleanupLocalDirs);
        Iterator it = this.executors.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            AppExecId fullId = (AppExecId)entry.getKey();
            ExecutorShuffleInfo executor = (ExecutorShuffleInfo)entry.getValue();
            if (!appId.equals(fullId.appId)) continue;
            it.remove();
            if (this.db != null) {
                try {
                    this.db.delete(ExternalShuffleBlockResolver.dbAppExecKey(fullId));
                }
                catch (IOException e) {
                    logger.error("Error deleting {} from executor state db", (Object)appId, (Object)e);
                }
            }
            if (!cleanupLocalDirs) continue;
            logger.info("Cleaning up executor {}'s {} local dirs", (Object)fullId, (Object)executor.localDirs.length);
            this.directoryCleaner.execute(() -> this.deleteExecutorDirs(executor.localDirs));
        }
    }

    public void executorRemoved(String executorId, String appId) {
        logger.info("Clean up non-shuffle and non-RDD files associated with the finished executor {}", (Object)executorId);
        AppExecId fullId = new AppExecId(appId, executorId);
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(fullId);
        if (executor == null) {
            logger.info("Executor is not registered (appId={}, execId={})", (Object)appId, (Object)executorId);
        } else {
            logger.info("Cleaning up non-shuffle and non-RDD files in executor {}'s {} local dirs", (Object)fullId, (Object)executor.localDirs.length);
            this.directoryCleaner.execute(() -> this.deleteNonShuffleServiceServedFiles(executor.localDirs));
        }
    }

    private void deleteExecutorDirs(String[] dirs) {
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively((File)new File(localDir));
                logger.debug("Successfully cleaned up directory: {}", (Object)localDir);
            }
            catch (Exception e) {
                logger.error("Failed to delete directory: " + localDir, (Throwable)e);
            }
        }
    }

    private void deleteNonShuffleServiceServedFiles(String[] dirs) {
        FilenameFilter filter = (dir, name) -> !name.endsWith(".index") && !name.endsWith(".data") && (!this.rddFetchEnabled || !name.startsWith("rdd_"));
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively((File)new File(localDir), (FilenameFilter)filter);
                logger.debug("Successfully cleaned up files not served by shuffle service in directory: {}", (Object)localDir);
            }
            catch (Exception e) {
                logger.error("Failed to delete files not served by shuffle service in directory: " + localDir, (Throwable)e);
            }
        }
    }

    private ManagedBuffer getSortBasedShuffleBlockData(ExecutorShuffleInfo executor, int shuffleId, long mapId, int startReduceId, int endReduceId) {
        File indexFile = ExecutorDiskUtils.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.index");
        try {
            ShuffleIndexInformation shuffleIndexInformation = (ShuffleIndexInformation)this.shuffleIndexCache.get((Object)indexFile);
            ShuffleIndexRecord shuffleIndexRecord = shuffleIndexInformation.getIndex(startReduceId, endReduceId);
            return new FileSegmentManagedBuffer(this.conf, ExecutorDiskUtils.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.data"), shuffleIndexRecord.getOffset(), shuffleIndexRecord.getLength());
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Failed to open file: " + indexFile, e);
        }
    }

    public ManagedBuffer getDiskPersistedRddBlockData(ExecutorShuffleInfo executor, int rddId, int splitIndex) {
        File file = ExecutorDiskUtils.getFile(executor.localDirs, executor.subDirsPerLocalDir, "rdd_" + rddId + "_" + splitIndex);
        long fileLength = file.length();
        FileSegmentManagedBuffer res = null;
        if (file.exists()) {
            res = new FileSegmentManagedBuffer(this.conf, file, 0L, fileLength);
        }
        return res;
    }

    void close() {
        if (this.db != null) {
            try {
                this.db.close();
            }
            catch (IOException e) {
                logger.error("Exception closing leveldb with registered executors", (Throwable)e);
            }
        }
    }

    public int removeBlocks(String appId, String execId, String[] blockIds) {
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        int numRemovedBlocks = 0;
        for (String blockId : blockIds) {
            File file = ExecutorDiskUtils.getFile(executor.localDirs, executor.subDirsPerLocalDir, blockId);
            if (file.delete()) {
                ++numRemovedBlocks;
                continue;
            }
            logger.warn("Failed to delete block: " + file.getAbsolutePath());
        }
        return numRemovedBlocks;
    }

    public Map<String, String[]> getLocalDirs(String appId, String[] execIds) {
        return Arrays.stream(execIds).map(exec -> {
            ExecutorShuffleInfo info = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, (String)exec));
            if (info == null) {
                throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, exec));
            }
            return Pair.of((Object)exec, (Object)info.localDirs);
        }).collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }

    private static byte[] dbAppExecKey(AppExecId appExecId) throws IOException {
        String appExecJson = mapper.writeValueAsString((Object)appExecId);
        String key = "AppExecShuffleInfo;" + appExecJson;
        return key.getBytes(StandardCharsets.UTF_8);
    }

    private static AppExecId parseDbAppExecKey(String s) throws IOException {
        if (!s.startsWith(APP_KEY_PREFIX)) {
            throw new IllegalArgumentException("expected a string starting with AppExecShuffleInfo");
        }
        String json = s.substring(APP_KEY_PREFIX.length() + 1);
        AppExecId parsed = (AppExecId)mapper.readValue(json, AppExecId.class);
        return parsed;
    }

    @VisibleForTesting
    static ConcurrentMap<AppExecId, ExecutorShuffleInfo> reloadRegisteredExecutors(DB db) throws IOException {
        ConcurrentMap registeredExecutors = Maps.newConcurrentMap();
        if (db != null) {
            Map.Entry e;
            String key;
            DBIterator itr = db.iterator();
            itr.seek(APP_KEY_PREFIX.getBytes(StandardCharsets.UTF_8));
            while (itr.hasNext() && (key = new String((byte[])(e = (Map.Entry)itr.next()).getKey(), StandardCharsets.UTF_8)).startsWith(APP_KEY_PREFIX)) {
                AppExecId id = ExternalShuffleBlockResolver.parseDbAppExecKey(key);
                logger.info("Reloading registered executors: " + id.toString());
                ExecutorShuffleInfo shuffleInfo = (ExecutorShuffleInfo)mapper.readValue((byte[])e.getValue(), ExecutorShuffleInfo.class);
                registeredExecutors.put(id, shuffleInfo);
            }
        }
        return registeredExecutors;
    }

    public static class AppExecId {
        public final String appId;
        public final String execId;

        @JsonCreator
        public AppExecId(@JsonProperty(value="appId") String appId, @JsonProperty(value="execId") String execId) {
            this.appId = appId;
            this.execId = execId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AppExecId appExecId = (AppExecId)o;
            return Objects.equals(this.appId, appExecId.appId) && Objects.equals(this.execId, appExecId.execId);
        }

        public int hashCode() {
            return Objects.hash(this.appId, this.execId);
        }

        public String toString() {
            return new ToStringBuilder((Object)this, ToStringStyle.SHORT_PREFIX_STYLE).append("appId", (Object)this.appId).append("execId", (Object)this.execId).toString();
        }
    }
}

