/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.utilint;

import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.utilint.IntStat;
import com.sleepycat.je.utilint.RateLimitingLogger;
import com.sleepycat.je.utilint.StatDefinition;
import com.sleepycat.je.utilint.StatGroup;
import java.io.Closeable;
import java.util.Date;
import java.util.Objects;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class TaskCoordinator {
    private final Set<Task> tasks;
    private final Set<Permit> outstandingPermits = ConcurrentHashMap.newKeySet();
    private final int maxRealPermits;
    private final AtomicInteger outstandingRealPermits = new AtomicInteger(0);
    private final AtomicInteger deficitPermits = new AtomicInteger(0);
    private volatile int appPermitPercent = 0;
    private final AtomicBoolean close = new AtomicBoolean(false);
    private final CoordinatorSemaphore permitSemaphore;
    private volatile TimerTask leaseCheckingTask;
    protected final Timer timer = new Timer(true);
    public static final int DEFAULT_LEASE_CHECK_PERIOD_MS = 1000;
    private static final int INACTIVE_REAL_PERMITS = 0x3FFFFFFF;
    protected final Logger logger;
    private final RateLimitingLogger<Task> deficitLogger;

    public TaskCoordinator(Logger logger, Set<Task> tasks, boolean active) {
        Objects.requireNonNull(logger, "logger argument must not be null");
        Objects.requireNonNull(tasks, "tasks argument must not be null");
        this.logger = logger;
        this.deficitLogger = new RateLimitingLogger(60000, tasks.size(), logger);
        this.tasks = tasks;
        int n = this.maxRealPermits = active ? tasks.stream().mapToInt(Task::getPermits).sum() : 0x3FFFFFFF;
        if (!tasks.isEmpty()) {
            logger.info("Coordinating tasks:" + tasks.stream().map(Task::getName).collect(Collectors.joining(", ")) + "Max real permits:" + this.maxRealPermits);
        }
        this.permitSemaphore = new CoordinatorSemaphore(this.maxRealPermits);
        this.setLeaseCheckingPeriod(1000);
        this.setAppPermitPercent(0);
    }

    public TaskCoordinator(Logger logger, Set<Task> tasks) {
        this(logger, tasks, true);
    }

    public void setLeaseCheckingPeriod(int periodMs) {
        if (this.leaseCheckingTask != null) {
            this.leaseCheckingTask.cancel();
            this.leaseCheckingTask = null;
        }
        if (periodMs == 0) {
            return;
        }
        this.leaseCheckingTask = new TimerTask(){

            @Override
            public void run() {
                for (Permit p : TaskCoordinator.this.outstandingPermits) {
                    if (!p.isExpired()) continue;
                    try {
                        p.close();
                    }
                    catch (IllegalStateException ise) {
                        TaskCoordinator.this.logger.warning("Detected (possibly leaked) permit with expired lease: " + p.getTask().getName() + " " + ise.getMessage());
                    }
                }
            }
        };
        this.timer.schedule(this.leaseCheckingTask, 0L, (long)periodMs);
    }

    public int getMaxPermits() {
        return this.maxRealPermits;
    }

    public int getAppPermits() {
        return this.maxRealPermits * this.appPermitPercent / 100;
    }

    public int getAppPermitPercent() {
        return this.appPermitPercent;
    }

    public int getAvailableRealPermits() {
        return this.permitSemaphore.availablePermits();
    }

    public int getOutstandingDeficitPermits() {
        return this.deficitPermits.get();
    }

    public int getOutstandingRealPermits() {
        return this.outstandingRealPermits.get();
    }

    public Permit acquirePermit(Task task, long timeout, long leaseInterval, TimeUnit unit) throws InterruptedException {
        Objects.requireNonNull(task, "task argument must not be null");
        Objects.requireNonNull(task, "unit argument must not be null");
        if (!this.tasks.contains(task)) {
            throw new IllegalArgumentException("Unknown task:" + task.getName());
        }
        long leaseIntervalMs = unit.toMillis(leaseInterval);
        if (this.close.get()) {
            return new DeficitPermit(task, leaseIntervalMs);
        }
        if (leaseInterval > 0L && leaseIntervalMs == 0L) {
            String msg = "Non-zero lease interval:" + leaseInterval + " " + unit.toString() + " must be >= 1 ms";
            throw new IllegalArgumentException(msg);
        }
        try {
            if (this.permitSemaphore.tryAcquire(1, timeout, unit)) {
                if (this.close.get()) {
                    return new DeficitPermit(task, leaseIntervalMs);
                }
                return new RealPermit(task, leaseIntervalMs);
            }
        }
        catch (InterruptedException iae) {
            this.logger.info("Permit acquisition for task:" + task.getName() + " was interrupted");
            throw iae;
        }
        this.deficitLogger.log(task, Level.INFO, "Granted deficit permit to " + task + " after waiting for " + timeout + " " + unit.toString() + ". Current app permit %: " + this.appPermitPercent);
        return new DeficitPermit(task, leaseIntervalMs);
    }

    public void releasePermit(Permit permit) {
        Objects.requireNonNull(permit, "permit argument must not be null");
        if (!this.tasks.contains(permit.getTask())) {
            throw new IllegalArgumentException("Unknown task:" + permit.getTask().getName());
        }
        permit.releasePermit();
    }

    public synchronized boolean setAppPermitPercent(int newAppPermitPercent) {
        if (this.appPermitPercent == newAppPermitPercent) {
            return false;
        }
        if (newAppPermitPercent < 0 || newAppPermitPercent > 100) {
            throw new IllegalArgumentException("Parameter must be a percentage:" + newAppPermitPercent);
        }
        int reqAppPermits = this.maxRealPermits * newAppPermitPercent / 100;
        int delta = reqAppPermits - this.getAppPermits();
        if (delta == 0) {
            this.appPermitPercent = newAppPermitPercent;
            return false;
        }
        if (delta > 0) {
            this.permitSemaphore.revoke(delta);
        } else {
            this.permitSemaphore.release(-delta);
        }
        this.appPermitPercent = newAppPermitPercent;
        return true;
    }

    public void close() {
        if (!this.close.compareAndSet(false, true)) {
            return;
        }
        this.logger.fine("Task Coordinator closed. " + this.permitSummary());
        this.permitSemaphore.release(0x3FFFFFFF);
        this.timer.cancel();
    }

    public StatGroup getStats(StatsConfig statsConfig) {
        StatGroup stats = new StatGroup("TaskCoordinator", "Task coordination ensures that the execution of background  housekeeping tasks is coordinated, so they do not all execute  at once.");
        new IntStat(stats, StatDefs.REAL_PERMITS, this.getOutstandingRealPermits());
        new IntStat(stats, StatDefs.DEFICIT_PERMITS, this.getOutstandingDeficitPermits());
        new IntStat(stats, StatDefs.APPLICATION_PERMITS, this.getAppPermits());
        return stats;
    }

    protected String permitSummary() {
        String taskSummary = this.tasks.stream().map(Task::toString).collect(Collectors.joining(", "));
        return String.format("App permits:%d%% (%d permits); Outstanding permits: %d real, %d deficit. %s", this.appPermitPercent, this.getAppPermits(), this.getOutstandingRealPermits(), this.getOutstandingDeficitPermits(), taskSummary);
    }

    public static interface StatDefs {
        public static final String GROUP_NAME = "TaskCoordinator";
        public static final String GROUP_DESC = "Task coordination ensures that the execution of background  housekeeping tasks is coordinated, so they do not all execute  at once.";
        public static final String REAL_PERMITS_NAME = "nRealPermits";
        public static final String REAL_PERMITS_DESC = "Number of real permits that have been currently granted to housekeeping tasks.";
        public static final StatDefinition REAL_PERMITS = new StatDefinition("nRealPermits", "Number of real permits that have been currently granted to housekeeping tasks.");
        public static final String DEFICIT_PERMITS_NAME = "nDeficitPermits";
        public static final String DEFICIT_PERMITS_DESC = "Number of deficit permits that have been currently granted to housekeeping tasks in the absence of real permits.";
        public static final StatDefinition DEFICIT_PERMITS = new StatDefinition("nDeficitPermits", "Number of deficit permits that have been currently granted to housekeeping tasks in the absence of real permits.");
        public static final String APPLICATION_PERMITS_NAME = "nApplicationPermits";
        public static final String APPLICATION_PERMITS_DESC = "Number of permits that have been currently reserved by the application and are therefor unavailable to housekeeping tasks.";
        public static final StatDefinition APPLICATION_PERMITS = new StatDefinition("nApplicationPermits", "Number of permits that have been currently reserved by the application and are therefor unavailable to housekeeping tasks.");
        public static final StatDefinition[] ALL = new StatDefinition[]{REAL_PERMITS, DEFICIT_PERMITS, APPLICATION_PERMITS};
    }

    public static class Task {
        final String name;
        final int permits;
        private final AtomicInteger realPermitsGranted = new AtomicInteger(0);
        private final AtomicInteger deficitPermitsGranted = new AtomicInteger(0);
        private final AtomicInteger expiredPermits = new AtomicInteger(0);

        public Task(String name, int permits) {
            Objects.requireNonNull(name, "name argument must not be null");
            this.name = name;
            this.permits = permits;
        }

        public String getName() {
            return this.name;
        }

        public int getPermits() {
            return this.permits;
        }

        public int getRealPermitsGranted() {
            return this.realPermitsGranted.get();
        }

        public void incRealPermits() {
            this.realPermitsGranted.incrementAndGet();
        }

        public int getDeficitPermitsGranted() {
            return this.deficitPermitsGranted.get();
        }

        public void incDeficitPermits() {
            this.deficitPermitsGranted.incrementAndGet();
        }

        public void incExpiredPermits() {
            this.expiredPermits.incrementAndGet();
        }

        public int getExpiredPermits() {
            return this.expiredPermits.get();
        }

        public void clearStats() {
            this.deficitPermitsGranted.set(0);
            this.realPermitsGranted.set(0);
        }

        public String toString() {
            return "< Task: " + this.name + ", permits:" + this.permits + " Real permits granted: " + this.realPermitsGranted + " Deficit permits granted: " + this.deficitPermitsGranted + " Expired permits:" + this.expiredPermits + " >";
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.name.hashCode();
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Task other = (Task)obj;
            return this.name.equals(other.name);
        }
    }

    private static class CoordinatorSemaphore
    extends Semaphore {
        CoordinatorSemaphore(int permits) {
            super(permits, true);
        }

        void revoke(int permits) {
            this.reducePermits(permits);
        }
    }

    private class DeficitPermit
    extends Permit {
        DeficitPermit(Task task, long leaseIntervalMs) {
            super(task, leaseIntervalMs);
            task.incDeficitPermits();
            TaskCoordinator.this.deficitPermits.incrementAndGet();
            TaskCoordinator.this.permitSemaphore.revoke(1);
        }

        @Override
        public boolean isDeficit() {
            return true;
        }

        @Override
        public synchronized void releasePermit() {
            super.releasePermit();
            TaskCoordinator.this.permitSemaphore.release(1);
            TaskCoordinator.this.deficitPermits.decrementAndGet();
            this.checkLeaseExpiry();
        }
    }

    private class RealPermit
    extends Permit {
        RealPermit(Task task, long leaseIntervalMs) {
            super(task, leaseIntervalMs);
            TaskCoordinator.this.outstandingRealPermits.incrementAndGet();
            task.incRealPermits();
        }

        @Override
        public boolean isDeficit() {
            return false;
        }

        @Override
        public synchronized void releasePermit() {
            super.releasePermit();
            TaskCoordinator.this.outstandingRealPermits.decrementAndGet();
            TaskCoordinator.this.permitSemaphore.release();
            this.checkLeaseExpiry();
        }
    }

    public abstract class Permit
    implements Closeable {
        private final Task task;
        private volatile long leaseEndMs = 0L;
        private volatile long releaseMs = 0L;

        public Permit(Task task, long leaseIntervalMs) {
            this.task = task;
            this.leaseEndMs = System.currentTimeMillis() + leaseIntervalMs;
            if (!TaskCoordinator.this.outstandingPermits.add(this)) {
                throw new IllegalStateException("Permit:" + task.getName() + " already present");
            }
        }

        Task getTask() {
            return this.task;
        }

        public synchronized boolean isReleased() {
            return this.releaseMs > 0L;
        }

        public synchronized boolean isExpired() {
            if (this.isReleased()) {
                return this.releaseMs > this.leaseEndMs;
            }
            return System.currentTimeMillis() > this.leaseEndMs;
        }

        protected synchronized void checkLeaseExpiry() {
            if (this.isExpired()) {
                throw new IllegalStateException("Permit expired at:" + new Date(this.leaseEndMs));
            }
        }

        private synchronized void checkReleased() {
            if (this.releaseMs > 0L) {
                String msg = "Permit for the task:'" + this.task.getName() + "' was previously released at " + new Date(this.releaseMs) + (this.isExpired() ? " Lease expired at " + new Date(this.leaseEndMs) : "");
                throw new IllegalStateException(msg);
            }
        }

        public synchronized void setLease(long leaseInterval, TimeUnit unit) {
            this.checkLeaseExpiry();
            this.checkReleased();
            this.leaseEndMs = System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(leaseInterval, unit);
        }

        public synchronized void releasePermit() {
            this.checkReleased();
            this.releaseMs = System.currentTimeMillis();
            if (this.isExpired()) {
                this.task.incExpiredPermits();
            }
            TaskCoordinator.this.outstandingPermits.remove(this);
        }

        public abstract boolean isDeficit();

        @Override
        public synchronized void close() {
            if (!this.isReleased()) {
                TaskCoordinator.this.releasePermit(this);
            }
        }
    }
}

