/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.DataInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.PrivateCellUtil;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.Reference;
import org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hadoop.hbase.regionserver.RegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSHDFSUtils;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.ServerRegionReplicaUtil;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class HRegionFileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(HRegionFileSystem.class);
    public static final String REGION_INFO_FILE = ".regioninfo";
    public static final String REGION_MERGES_DIR = ".merges";
    public static final String REGION_SPLITS_DIR = ".splits";
    @VisibleForTesting
    static final String REGION_TEMP_DIR = ".tmp";
    private final RegionInfo regionInfo;
    final RegionInfo regionInfoForFs;
    final Configuration conf;
    private final Path tableDir;
    final FileSystem fs;
    private final Path regionDir;
    private final int hdfsClientRetriesNumber;
    private final int baseSleepBeforeRetries;
    private static final int DEFAULT_HDFS_CLIENT_RETRIES_NUMBER = 10;
    private static final int DEFAULT_BASE_SLEEP_BEFORE_RETRIES = 1000;

    HRegionFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo) {
        this.fs = fs;
        this.conf = conf;
        this.tableDir = Objects.requireNonNull(tableDir, "tableDir is null");
        this.regionInfo = Objects.requireNonNull(regionInfo, "regionInfo is null");
        this.regionInfoForFs = ServerRegionReplicaUtil.getRegionInfoForFs(regionInfo);
        this.regionDir = FSUtils.getRegionDirFromTableDir(tableDir, regionInfo);
        this.hdfsClientRetriesNumber = conf.getInt("hdfs.client.retries.number", 10);
        this.baseSleepBeforeRetries = conf.getInt("hdfs.client.sleep.before.retries", 1000);
    }

    public FileSystem getFileSystem() {
        return this.fs;
    }

    public RegionInfo getRegionInfo() {
        return this.regionInfo;
    }

    public RegionInfo getRegionInfoForFS() {
        return this.regionInfoForFs;
    }

    public Path getTableDir() {
        return this.tableDir;
    }

    public Path getRegionDir() {
        return this.regionDir;
    }

    Path getTempDir() {
        return new Path(this.getRegionDir(), REGION_TEMP_DIR);
    }

    void cleanupTempDir() throws IOException {
        this.deleteDir(this.getTempDir());
    }

    public Path getStoreDir(String familyName) {
        return new Path(this.getRegionDir(), familyName);
    }

    Path createStoreDir(String familyName) throws IOException {
        Path storeDir = this.getStoreDir(familyName);
        if (!this.fs.exists(storeDir) && !this.createDir(storeDir)) {
            throw new IOException("Failed creating " + storeDir);
        }
        return storeDir;
    }

    public void setStoragePolicy(String familyName, String policyName) {
        FSUtils.setStoragePolicy(this.fs, this.getStoreDir(familyName), policyName);
    }

    @Nullable
    public String getStoragePolicyName(String familyName) {
        if (this.fs instanceof HFileSystem) {
            Path storeDir = this.getStoreDir(familyName);
            return ((HFileSystem)this.fs).getStoragePolicyName(storeDir);
        }
        return null;
    }

    public Collection<StoreFileInfo> getStoreFiles(byte[] familyName) throws IOException {
        return this.getStoreFiles(Bytes.toString(familyName));
    }

    public Collection<StoreFileInfo> getStoreFiles(String familyName) throws IOException {
        return this.getStoreFiles(familyName, true);
    }

    public Collection<StoreFileInfo> getStoreFiles(String familyName, boolean validate) throws IOException {
        Path familyDir = this.getStoreDir(familyName);
        FileStatus[] files = FSUtils.listStatus(this.fs, familyDir);
        if (files == null) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("No StoreFiles for: " + familyDir);
            }
            return null;
        }
        ArrayList<StoreFileInfo> storeFiles = new ArrayList<StoreFileInfo>(files.length);
        for (FileStatus status : files) {
            if (validate && !StoreFileInfo.isValid(status)) {
                LOG.warn("Invalid StoreFile: " + status.getPath());
                continue;
            }
            StoreFileInfo info = ServerRegionReplicaUtil.getStoreFileInfo(this.conf, this.fs, this.regionInfo, this.regionInfoForFs, familyName, status.getPath());
            storeFiles.add(info);
        }
        return storeFiles;
    }

    public static List<LocatedFileStatus> getStoreFilesLocatedStatus(HRegionFileSystem regionfs, String familyName, boolean validate) throws IOException {
        Path familyDir = regionfs.getStoreDir(familyName);
        List<LocatedFileStatus> locatedFileStatuses = FSUtils.listLocatedStatus(regionfs.getFileSystem(), familyDir);
        if (locatedFileStatuses == null) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("No StoreFiles for: " + familyDir);
            }
            return null;
        }
        ArrayList<LocatedFileStatus> validStoreFiles = Lists.newArrayList();
        for (LocatedFileStatus status : locatedFileStatuses) {
            if (validate && !StoreFileInfo.isValid((FileStatus)status)) {
                LOG.warn("Invalid StoreFile: " + status.getPath());
                continue;
            }
            validStoreFiles.add(status);
        }
        return validStoreFiles;
    }

    Path getStoreFilePath(String familyName, String fileName) {
        Path familyDir = this.getStoreDir(familyName);
        return new Path(familyDir, fileName).makeQualified(this.fs.getUri(), this.fs.getWorkingDirectory());
    }

    StoreFileInfo getStoreFileInfo(String familyName, String fileName) throws IOException {
        Path familyDir = this.getStoreDir(familyName);
        return ServerRegionReplicaUtil.getStoreFileInfo(this.conf, this.fs, this.regionInfo, this.regionInfoForFs, familyName, new Path(familyDir, fileName));
    }

    public boolean hasReferences(String familyName) throws IOException {
        Path storeDir = this.getStoreDir(familyName);
        FileStatus[] files = FSUtils.listStatus(this.fs, storeDir);
        if (files != null) {
            for (FileStatus stat : files) {
                if (stat.isDirectory() || !StoreFileInfo.isReference(stat.getPath())) continue;
                LOG.trace("Reference {}", (Object)stat.getPath());
                return true;
            }
        }
        return false;
    }

    public boolean hasReferences(TableDescriptor htd) throws IOException {
        for (ColumnFamilyDescriptor family : htd.getColumnFamilies()) {
            if (!this.hasReferences(family.getNameAsString())) continue;
            return true;
        }
        return false;
    }

    public Collection<String> getFamilies() throws IOException {
        FileStatus[] fds = FSUtils.listStatus(this.fs, this.getRegionDir(), new FSUtils.FamilyDirFilter(this.fs));
        if (fds == null) {
            return null;
        }
        ArrayList<String> families = new ArrayList<String>(fds.length);
        for (FileStatus status : fds) {
            families.add(status.getPath().getName());
        }
        return families;
    }

    public void deleteFamily(String familyName) throws IOException {
        HFileArchiver.archiveFamily(this.fs, this.conf, this.regionInfoForFs, this.tableDir, Bytes.toBytes(familyName));
        Path familyDir = this.getStoreDir(familyName);
        if (this.fs.exists(familyDir) && !this.deleteDir(familyDir)) {
            throw new IOException("Could not delete family " + familyName + " from FileSystem for region " + this.regionInfoForFs.getRegionNameAsString() + "(" + this.regionInfoForFs.getEncodedName() + ")");
        }
    }

    private static String generateUniqueName(String suffix) {
        String name = UUID.randomUUID().toString().replaceAll("-", "");
        if (suffix != null) {
            name = name + suffix;
        }
        return name;
    }

    public Path createTempName() {
        return this.createTempName(null);
    }

    public Path createTempName(String suffix) {
        return new Path(this.getTempDir(), HRegionFileSystem.generateUniqueName(suffix));
    }

    public Path commitStoreFile(String familyName, Path buildPath) throws IOException {
        Path dstPath = this.preCommitStoreFile(familyName, buildPath, -1L, false);
        return this.commitStoreFile(buildPath, dstPath);
    }

    private Path preCommitStoreFile(String familyName, Path buildPath, long seqNum, boolean generateNewName) throws IOException {
        Path storeDir = this.getStoreDir(familyName);
        if (!this.fs.exists(storeDir) && !this.createDir(storeDir)) {
            throw new IOException("Failed creating " + storeDir);
        }
        String name = buildPath.getName();
        if (generateNewName) {
            name = HRegionFileSystem.generateUniqueName(seqNum < 0L ? null : "_SeqId_" + seqNum + "_");
        }
        Path dstPath = new Path(storeDir, name);
        if (!this.fs.exists(buildPath)) {
            throw new FileNotFoundException(buildPath.toString());
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Committing " + buildPath + " as " + dstPath);
        }
        return dstPath;
    }

    Path commitStoreFile(Path buildPath, Path dstPath) throws IOException {
        if (!this.rename(buildPath, dstPath)) {
            throw new IOException("Failed rename of " + buildPath + " to " + dstPath);
        }
        return dstPath;
    }

    public void removeStoreFile(String familyName, Path filePath) throws IOException {
        HFileArchiver.archiveStoreFile(this.conf, this.fs, this.regionInfoForFs, this.tableDir, Bytes.toBytes(familyName), filePath);
    }

    public void removeStoreFiles(String familyName, Collection<HStoreFile> storeFiles) throws IOException {
        HFileArchiver.archiveStoreFiles(this.conf, this.fs, this.regionInfoForFs, this.tableDir, Bytes.toBytes(familyName), storeFiles);
    }

    Pair<Path, Path> bulkLoadStoreFile(String familyName, Path srcPath, long seqNum) throws IOException {
        FileSystem desFs;
        FileSystem srcFs = srcPath.getFileSystem(this.conf);
        srcPath = srcFs.resolvePath(srcPath);
        FileSystem realSrcFs = srcPath.getFileSystem(this.conf);
        FileSystem fileSystem = desFs = this.fs instanceof HFileSystem ? ((HFileSystem)this.fs).getBackingFs() : this.fs;
        if (!FSHDFSUtils.isSameHdfs(this.conf, realSrcFs, desFs)) {
            LOG.info("Bulk-load file " + srcPath + " is on different filesystem than the destination store. Copying file over to destination filesystem.");
            Path tmpPath = this.createTempName();
            FileUtil.copy((FileSystem)realSrcFs, (Path)srcPath, (FileSystem)this.fs, (Path)tmpPath, (boolean)false, (Configuration)this.conf);
            LOG.info("Copied " + srcPath + " to temporary path on destination filesystem: " + tmpPath);
            srcPath = tmpPath;
        }
        return new Pair<Path, Path>(srcPath, this.preCommitStoreFile(familyName, srcPath, seqNum, true));
    }

    Path getSplitsDir() {
        return new Path(this.getRegionDir(), REGION_SPLITS_DIR);
    }

    public Path getSplitsDir(RegionInfo hri) {
        return new Path(this.getSplitsDir(), hri.getEncodedName());
    }

    void cleanupSplitsDir() throws IOException {
        this.deleteDir(this.getSplitsDir());
    }

    void cleanupAnySplitDetritus() throws IOException {
        Path splitdir = this.getSplitsDir();
        if (!this.fs.exists(splitdir)) {
            return;
        }
        FileStatus[] daughters = FSUtils.listStatus(this.fs, splitdir, new FSUtils.DirFilter(this.fs));
        if (daughters != null) {
            for (FileStatus daughter : daughters) {
                Path daughterDir = new Path(this.getTableDir(), daughter.getPath().getName());
                if (!this.fs.exists(daughterDir) || this.deleteDir(daughterDir)) continue;
                throw new IOException("Failed delete of " + daughterDir);
            }
        }
        this.cleanupSplitsDir();
        LOG.info("Cleaned up old failed split transaction detritus: " + splitdir);
    }

    void cleanupDaughterRegion(RegionInfo regionInfo) throws IOException {
        Path regionDir = new Path(this.tableDir, regionInfo.getEncodedName());
        if (this.fs.exists(regionDir) && !this.deleteDir(regionDir)) {
            throw new IOException("Failed delete of " + regionDir);
        }
    }

    public Path commitDaughterRegion(RegionInfo regionInfo) throws IOException {
        Path regionDir = new Path(this.tableDir, regionInfo.getEncodedName());
        Path daughterTmpDir = this.getSplitsDir(regionInfo);
        if (this.fs.exists(daughterTmpDir)) {
            Path regionInfoFile = new Path(daughterTmpDir, REGION_INFO_FILE);
            byte[] regionInfoContent = HRegionFileSystem.getRegionInfoFileContent(regionInfo);
            HRegionFileSystem.writeRegionInfoFileContent(this.conf, this.fs, regionInfoFile, regionInfoContent);
            if (!this.rename(daughterTmpDir, regionDir)) {
                throw new IOException("Unable to rename " + daughterTmpDir + " to " + regionDir);
            }
        }
        return regionDir;
    }

    public void createSplitsDir(RegionInfo daughterA, RegionInfo daughterB) throws IOException {
        Path splitdir = this.getSplitsDir();
        if (this.fs.exists(splitdir)) {
            LOG.info("The " + splitdir + " directory exists.  Hence deleting it to recreate it");
            if (!this.deleteDir(splitdir)) {
                throw new IOException("Failed deletion of " + splitdir + " before creating them again.");
            }
        }
        if (!this.createDir(splitdir)) {
            throw new IOException("Failed create of " + splitdir);
        }
        Path daughterATmpDir = this.getSplitsDir(daughterA);
        if (!this.createDir(daughterATmpDir)) {
            throw new IOException("Failed create of " + daughterATmpDir);
        }
        Path daughterBTmpDir = this.getSplitsDir(daughterB);
        if (!this.createDir(daughterBTmpDir)) {
            throw new IOException("Failed create of " + daughterBTmpDir);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Unable to fully structure code
     */
    public Path splitStoreFile(RegionInfo hri, String familyName, HStoreFile f, byte[] splitRow, boolean top, RegionSplitPolicy splitPolicy) throws IOException {
        block9: {
            block7: {
                block8: {
                    block6: {
                        block5: {
                            if (splitPolicy != null && splitPolicy.skipStoreFileRangeCheck(familyName)) break block9;
                            f.initReader();
                            try {
                                if (!top) ** GOTO lbl23
                                splitKey = PrivateCellUtil.createFirstOnRow(splitRow);
                                lastKey = f.getLastKey();
                                if (lastKey.isPresent()) break block5;
                                var9_9 = null;
                                f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
                            }
                            catch (Throwable var10_14) {
                                f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
                                throw var10_14;
                            }
                            return var9_9;
                        }
                        if (f.getComparator().compare(splitKey, lastKey.get()) <= 0) break block6;
                        var9_10 = null;
                        f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
                        return var9_10;
                    }
                    break block7;
lbl23:
                    // 1 sources

                    splitKey = PrivateCellUtil.createLastOnRow(splitRow);
                    firstKey = f.getFirstKey();
                    if (firstKey.isPresent()) break block8;
                    var9_11 = null;
                    f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
                    return var9_11;
                }
                if (f.getComparator().compare(splitKey, firstKey.get()) >= 0) break block7;
                var9_12 = null;
                f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
                return var9_12;
            }
            f.closeStoreFile(f.getCacheConf() != null ? f.getCacheConf().shouldEvictOnClose() : true);
        }
        splitDir = new Path(this.getSplitsDir(hri), familyName);
        r = top != false ? Reference.createTopReference(splitRow) : Reference.createBottomReference(splitRow);
        parentRegionName = this.regionInfoForFs.getEncodedName();
        p = new Path(splitDir, f.getPath().getName() + "." + parentRegionName);
        return r.write(this.fs, p);
    }

    public Path getMergesDir() {
        return new Path(this.getRegionDir(), REGION_MERGES_DIR);
    }

    Path getMergesDir(RegionInfo hri) {
        return new Path(this.getMergesDir(), hri.getEncodedName());
    }

    void cleanupMergesDir() throws IOException {
        this.deleteDir(this.getMergesDir());
    }

    public void cleanupMergedRegion(RegionInfo mergedRegion) throws IOException {
        Path regionDir = new Path(this.tableDir, mergedRegion.getEncodedName());
        if (this.fs.exists(regionDir) && !this.fs.delete(regionDir, true)) {
            throw new IOException("Failed delete of " + regionDir);
        }
    }

    static boolean mkdirs(FileSystem fs, Configuration conf, Path dir) throws IOException {
        if (FSUtils.isDistributedFileSystem(fs) || !conf.getBoolean("hbase.data.umask.enable", false)) {
            return fs.mkdirs(dir);
        }
        FsPermission perms = FSUtils.getFilePermissions(fs, conf, "hbase.data.umask");
        return fs.mkdirs(dir, perms);
    }

    public void createMergesDir() throws IOException {
        Path mergesdir = this.getMergesDir();
        if (this.fs.exists(mergesdir)) {
            LOG.info("{} directory exists. Deleting it to recreate it anew", (Object)mergesdir);
            if (!this.fs.delete(mergesdir, true)) {
                throw new IOException("Failed deletion of " + mergesdir + " before recreate.");
            }
        }
        if (!HRegionFileSystem.mkdirs(this.fs, this.conf, mergesdir)) {
            throw new IOException("Failed create of " + mergesdir);
        }
    }

    public Path mergeStoreFile(RegionInfo mergedRegion, String familyName, HStoreFile f, Path mergedDir) throws IOException {
        Path referenceDir = new Path(new Path(mergedDir, mergedRegion.getEncodedName()), familyName);
        Reference r = Reference.createTopReference(this.regionInfoForFs.getStartKey());
        String mergingRegionName = this.regionInfoForFs.getEncodedName();
        Path p = new Path(referenceDir, f.getPath().getName() + "." + mergingRegionName);
        return r.write(this.fs, p);
    }

    public void commitMergedRegion(RegionInfo mergedRegionInfo) throws IOException {
        Path regionDir = new Path(this.tableDir, mergedRegionInfo.getEncodedName());
        Path mergedRegionTmpDir = this.getMergesDir(mergedRegionInfo);
        if (mergedRegionTmpDir != null && this.fs.exists(mergedRegionTmpDir) && !this.fs.rename(mergedRegionTmpDir, regionDir)) {
            throw new IOException("Unable to rename " + mergedRegionTmpDir + " to " + regionDir);
        }
    }

    void logFileSystemState(Logger LOG) throws IOException {
        FSUtils.logFileSystemState(this.fs, this.getRegionDir(), LOG);
    }

    private static byte[] getRegionInfoFileContent(RegionInfo hri) throws IOException {
        return RegionInfo.toDelimitedByteArray(hri);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static RegionInfo loadRegionInfoFileContent(FileSystem fs, Path regionDir) throws IOException {
        try (FSDataInputStream in = fs.open(new Path(regionDir, REGION_INFO_FILE));){
            RegionInfo regionInfo = RegionInfo.parseFrom((DataInputStream)in);
            return regionInfo;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeRegionInfoFileContent(Configuration conf, FileSystem fs, Path regionInfoFile, byte[] content) throws IOException {
        FsPermission perms = FSUtils.getFilePermissions(fs, conf, "hbase.data.umask");
        try (FSDataOutputStream out = FSUtils.create(conf, fs, regionInfoFile, perms, null);){
            out.write(content);
        }
    }

    void checkRegionInfoOnFilesystem() throws IOException {
        byte[] content = HRegionFileSystem.getRegionInfoFileContent(this.regionInfoForFs);
        try {
            FileStatus fileStatus = this.fs.getFileStatus(this.getRegionDir());
        }
        catch (FileNotFoundException e) {
            LOG.warn(this.getRegionDir() + " doesn't exist for region: " + this.regionInfoForFs.getEncodedName() + " on table " + this.regionInfo.getTable());
        }
        try {
            Path regionInfoFile = new Path(this.getRegionDir(), REGION_INFO_FILE);
            FileStatus status = this.fs.getFileStatus(regionInfoFile);
            if (status != null && status.getLen() == (long)content.length) {
                return;
            }
            LOG.info("Rewriting .regioninfo file at: " + regionInfoFile);
            if (!this.fs.delete(regionInfoFile, false)) {
                throw new IOException("Unable to remove existing " + regionInfoFile);
            }
        }
        catch (FileNotFoundException e) {
            LOG.warn(".regioninfo file not found for region: " + this.regionInfoForFs.getEncodedName() + " on table " + this.regionInfo.getTable());
        }
        this.writeRegionInfoOnFilesystem(content, true);
    }

    private void writeRegionInfoOnFilesystem(boolean useTempDir) throws IOException {
        byte[] content = HRegionFileSystem.getRegionInfoFileContent(this.regionInfoForFs);
        this.writeRegionInfoOnFilesystem(content, useTempDir);
    }

    private void writeRegionInfoOnFilesystem(byte[] regionInfoContent, boolean useTempDir) throws IOException {
        Path regionInfoFile = new Path(this.getRegionDir(), REGION_INFO_FILE);
        if (useTempDir) {
            Path tmpPath = new Path(this.getTempDir(), REGION_INFO_FILE);
            if (FSUtils.isExists(this.fs, tmpPath)) {
                FSUtils.delete(this.fs, tmpPath, true);
            }
            HRegionFileSystem.writeRegionInfoFileContent(this.conf, this.fs, tmpPath, regionInfoContent);
            if (this.fs.exists(tmpPath) && !this.rename(tmpPath, regionInfoFile)) {
                throw new IOException("Unable to rename " + tmpPath + " to " + regionInfoFile);
            }
        } else {
            HRegionFileSystem.writeRegionInfoFileContent(this.conf, this.fs, regionInfoFile, regionInfoContent);
        }
    }

    public static HRegionFileSystem createRegionOnFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        if (regionInfo.getReplicaId() == 0) {
            Path regionDir = regionFs.getRegionDir();
            if (fs.exists(regionDir)) {
                LOG.warn("Trying to create a region that already exists on disk: " + regionDir);
            } else if (!HRegionFileSystem.createDirOnFileSystem(fs, conf, regionDir)) {
                LOG.warn("Unable to create the region directory: " + regionDir);
                throw new IOException("Unable to create region directory: " + regionDir);
            }
            regionFs.writeRegionInfoOnFilesystem(false);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Skipping creation of .regioninfo file for " + regionInfo);
        }
        return regionFs;
    }

    public static HRegionFileSystem openRegionFromFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo, boolean readOnly) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        Path regionDir = regionFs.getRegionDir();
        if (!fs.exists(regionDir)) {
            LOG.warn("Trying to open a region that do not exists on disk: " + regionDir);
            throw new IOException("The specified region do not exists on disk: " + regionDir);
        }
        if (!readOnly) {
            regionFs.cleanupTempDir();
            regionFs.cleanupSplitsDir();
            regionFs.cleanupMergesDir();
            if (regionInfo.getReplicaId() == 0) {
                regionFs.checkRegionInfoOnFilesystem();
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("Skipping creation of .regioninfo file for " + regionInfo);
            }
        }
        return regionFs;
    }

    public static void deleteRegionFromFileSystem(Configuration conf, FileSystem fs, Path tableDir, RegionInfo regionInfo) throws IOException {
        HRegionFileSystem regionFs = new HRegionFileSystem(conf, fs, tableDir, regionInfo);
        Path regionDir = regionFs.getRegionDir();
        if (!fs.exists(regionDir)) {
            LOG.warn("Trying to delete a region that do not exists on disk: " + regionDir);
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("DELETING region " + regionDir);
        }
        Path rootDir = FSUtils.getRootDir(conf);
        HFileArchiver.archiveRegion(fs, rootDir, tableDir, regionDir);
        if (!fs.delete(regionDir, true)) {
            LOG.warn("Failed delete of " + regionDir);
        }
    }

    boolean createDir(Path dir) throws IOException {
        int i = 0;
        IOException lastIOE = null;
        while (true) {
            try {
                return HRegionFileSystem.mkdirs(this.fs, this.conf, dir);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (this.fs.exists(dir)) {
                    return true;
                }
                try {
                    this.sleepBeforeRetry("Create Directory", i + 1);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in createDir", lastIOE);
            }
            break;
        }
    }

    boolean rename(Path srcpath, Path dstPath) throws IOException {
        IOException lastIOE = null;
        int i = 0;
        while (true) {
            try {
                return this.fs.rename(srcpath, dstPath);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (!this.fs.exists(srcpath) && this.fs.exists(dstPath)) {
                    return true;
                }
                try {
                    this.sleepBeforeRetry("Rename Directory", i + 1);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in rename", lastIOE);
            }
            break;
        }
    }

    boolean deleteDir(Path dir) throws IOException {
        IOException lastIOE = null;
        int i = 0;
        while (true) {
            try {
                return this.fs.delete(dir, true);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (!this.fs.exists(dir)) {
                    return true;
                }
                try {
                    this.sleepBeforeRetry("Delete Directory", i + 1);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= this.hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in DeleteDir", lastIOE);
            }
            break;
        }
    }

    private void sleepBeforeRetry(String msg, int sleepMultiplier) throws InterruptedException {
        HRegionFileSystem.sleepBeforeRetry(msg, sleepMultiplier, this.baseSleepBeforeRetries, this.hdfsClientRetriesNumber);
    }

    private static boolean createDirOnFileSystem(FileSystem fs, Configuration conf, Path dir) throws IOException {
        int i = 0;
        IOException lastIOE = null;
        int hdfsClientRetriesNumber = conf.getInt("hdfs.client.retries.number", 10);
        int baseSleepBeforeRetries = conf.getInt("hdfs.client.sleep.before.retries", 1000);
        while (true) {
            try {
                return fs.mkdirs(dir);
            }
            catch (IOException ioe) {
                lastIOE = ioe;
                if (fs.exists(dir)) {
                    return true;
                }
                try {
                    HRegionFileSystem.sleepBeforeRetry("Create Directory", i + 1, baseSleepBeforeRetries, hdfsClientRetriesNumber);
                    continue;
                }
                catch (InterruptedException e) {
                    throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                }
                if (++i <= hdfsClientRetriesNumber) continue;
                throw new IOException("Exception in createDir", lastIOE);
            }
            break;
        }
    }

    private static void sleepBeforeRetry(String msg, int sleepMultiplier, int baseSleepBeforeRetries, int hdfsClientRetriesNumber) throws InterruptedException {
        if (sleepMultiplier > hdfsClientRetriesNumber) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(msg + ", retries exhausted");
            }
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug(msg + ", sleeping " + baseSleepBeforeRetries + " times " + sleepMultiplier);
        }
        Thread.sleep((long)baseSleepBeforeRetries * (long)sleepMultiplier);
    }
}

