/*
 * Decompiled with CFR 0.152.
 */
package org.irods.irods4j.high_level.connection;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.irods.irods4j.common.Versioning;
import org.irods.irods4j.high_level.catalog.IRODSQuery;
import org.irods.irods4j.high_level.connection.ConnectionPoolOptions;
import org.irods.irods4j.high_level.connection.IRODSConnection;
import org.irods.irods4j.high_level.connection.QualifiedUsername;
import org.irods.irods4j.low_level.api.IRODSApi;
import org.irods.irods4j.low_level.api.IRODSException;

public class IRODSConnectionPool
implements AutoCloseable {
    private static final Logger log = LogManager.getLogger();
    private IRODSApi.ConnectionOptions connOptions;
    private ConnectionPoolOptions poolOptions;
    private int poolSize;
    private String host;
    private int port;
    private QualifiedUsername clientUser;
    private Function<IRODSApi.RcComm, Boolean> authenticator;
    private List<ConnectionContext> pool;

    public IRODSConnectionPool(int poolSize) {
        IRODSConnectionPool.throwIfLessThanOrEqualTo(poolSize, 0, "Connection pool size is less than or equal to 0");
        this.doConstructor(new IRODSApi.ConnectionOptions(), new ConnectionPoolOptions(), poolSize);
    }

    public IRODSConnectionPool(ConnectionPoolOptions poolOptions, int poolSize) {
        IRODSConnectionPool.throwIfNull(poolOptions, "Connection pool options is null");
        IRODSConnectionPool.throwIfLessThanOrEqualTo(poolSize, 0, "Connection pool size is less than or equal to 0");
        this.doConstructor(new IRODSApi.ConnectionOptions(), poolOptions, poolSize);
    }

    public IRODSConnectionPool(IRODSApi.ConnectionOptions connOptions, int poolSize) {
        IRODSConnectionPool.throwIfNull(connOptions, "Connection options is null");
        IRODSConnectionPool.throwIfLessThanOrEqualTo(poolSize, 0, "Connection pool size is less than or equal to 0");
        this.doConstructor(connOptions, new ConnectionPoolOptions(), poolSize);
    }

    public IRODSConnectionPool(IRODSApi.ConnectionOptions connOptions, ConnectionPoolOptions poolOptions, int poolSize) {
        IRODSConnectionPool.throwIfNull(connOptions, "Connection options is null");
        IRODSConnectionPool.throwIfLessThanOrEqualTo(poolSize, 0, "Connection pool size is less than or equal to 0");
        this.doConstructor(connOptions, poolOptions, poolSize);
    }

    public void setConnectionOptions(IRODSApi.ConnectionOptions connOptions) {
        IRODSConnectionPool.throwIfNull(connOptions, "Connection options is null");
        this.connOptions = connOptions;
    }

    public void setPoolSize(int poolSize) {
        IRODSConnectionPool.throwIfLessThanOrEqualTo(poolSize, 0, "Connection pool size is less than or equal to 0");
        this.poolSize = poolSize;
    }

    public int getPoolSize() {
        return this.pool.size();
    }

    public void start(String host, int port, QualifiedUsername clientUser, Function<IRODSApi.RcComm, Boolean> authenticator) throws IOException, IRODSException {
        IRODSConnectionPool.throwIfInvalidHost(host);
        IRODSConnectionPool.throwIfInvalidPortNumber(port);
        IRODSConnectionPool.throwIfInvalidClientUser(clientUser);
        this.host = host;
        this.port = port;
        this.clientUser = clientUser;
        this.authenticator = authenticator;
        this.doStart(Optional.empty());
    }

    public void start(ExecutorService executor, String host, int port, QualifiedUsername clientUser, Function<IRODSApi.RcComm, Boolean> authenticator) throws IOException, IRODSException {
        IRODSConnectionPool.throwIfNull(executor, "Executor service is null");
        IRODSConnectionPool.throwIfInvalidHost(host);
        IRODSConnectionPool.throwIfInvalidPortNumber(port);
        IRODSConnectionPool.throwIfInvalidClientUser(clientUser);
        this.host = host;
        this.port = port;
        this.clientUser = clientUser;
        this.authenticator = authenticator;
        this.doStart(Optional.of(executor));
    }

    public void stop() {
        this.pool.forEach(ctx -> {
            try {
                if (null != ctx.conn) {
                    ctx.conn.disconnect();
                }
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        });
    }

    @Override
    public void close() throws Exception {
        this.stop();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PoolConnection getConnection() {
        int i = 0;
        while (true) {
            ConnectionContext ctx = this.pool.get(i);
            if (ctx.lock.tryLock()) {
                try {
                    if (!ctx.inUse.get()) {
                        ctx.inUse.set(true);
                        this.refreshConnection(ctx);
                        if (this.poolOptions.numberOfRetrievalsBeforeConnectionRefresh.isPresent()) {
                            ++ctx.retrievalCount;
                        }
                        PoolConnection poolConnection = new PoolConnection(ctx);
                        return poolConnection;
                    }
                }
                finally {
                    ctx.lock.unlock();
                }
            }
            i = (i + 1) % this.pool.size();
        }
    }

    private static void throwIfNull(Object object, String msg) {
        if (null == object) {
            throw new IllegalArgumentException(msg);
        }
    }

    private static void throwIfInvalidHost(String host) {
        if (null == host || host.isEmpty()) {
            throw new IllegalArgumentException("Host is null or empty");
        }
    }

    private static void throwIfInvalidPortNumber(int port) {
        if (port <= 0) {
            throw new IllegalArgumentException("Port is less than or equal to 0");
        }
    }

    private static void throwIfInvalidClientUser(QualifiedUsername user) {
        if (null == user) {
            throw new IllegalArgumentException("Client user is null");
        }
    }

    private static void throwIfLessThanOrEqualTo(int value, int lowerBound, String msg) {
        if (value <= lowerBound) {
            throw new IllegalArgumentException(msg);
        }
    }

    private void doConstructor(IRODSApi.ConnectionOptions connOptions, ConnectionPoolOptions poolOptions, int poolSize) {
        poolOptions.numberOfRetrievalsBeforeConnectionRefresh.ifPresent(v -> IRODSConnectionPool.throwIfLessThanOrEqualTo(v, 0, "Connection pool option [numberOfRetrievalsBeforeConnectionRefresh] is less than or equal to 0"));
        poolOptions.numberOfSecondsBeforeConnectionRefresh.ifPresent(v -> IRODSConnectionPool.throwIfLessThanOrEqualTo(v, 0, "Connection pool option [numberOfSecondsBeforeConnectionRefresh] is less than or equal to 0"));
        this.connOptions = connOptions;
        this.poolOptions = poolOptions;
        this.poolSize = poolSize;
        this.pool = new ArrayList<ConnectionContext>(poolSize);
    }

    private void doStart(Optional<ExecutorService> executor) throws IOException, IRODSException {
        this.pool.clear();
        for (int i = 0; i < this.poolSize; ++i) {
            this.pool.add(new ConnectionContext());
        }
        AtomicBoolean connectFailed = new AtomicBoolean();
        AtomicBoolean authFailed = new AtomicBoolean();
        if (executor.isPresent()) {
            ArrayList futures = new ArrayList();
            this.pool.forEach(ctx -> futures.add(((ExecutorService)executor.get()).submit(() -> {
                IRODSConnection conn = new IRODSConnection(this.connOptions);
                try {
                    conn.connect(this.host, this.port, this.clientUser);
                }
                catch (Exception e) {
                    connectFailed.set(true);
                    return;
                }
                if (!this.authenticator.apply(conn.getRcComm()).booleanValue()) {
                    authFailed.set(true);
                    try {
                        conn.disconnect();
                    }
                    catch (IOException e) {
                        log.debug(e.getMessage());
                    }
                    return;
                }
                ctx.conn = conn;
                log.debug("Connection established with iRODS server [host={}, port={}].", (Object)this.host, (Object)this.port);
            })));
            for (Future f : futures) {
                try {
                    f.get();
                }
                catch (InterruptedException | ExecutionException exception) {}
            }
        } else {
            for (int i = 0; i < this.pool.size(); ++i) {
                IRODSConnection conn = new IRODSConnection(this.connOptions);
                try {
                    conn.connect(this.host, this.port, this.clientUser);
                }
                catch (Exception e) {
                    connectFailed.set(true);
                    log.error(e.getMessage());
                    break;
                }
                if (!this.authenticator.apply(conn.getRcComm()).booleanValue()) {
                    authFailed.set(true);
                    try {
                        conn.disconnect();
                    }
                    catch (IOException e) {
                        log.debug(e.getMessage());
                    }
                    break;
                }
                this.pool.get((int)i).conn = conn;
                log.debug("Connection established with iRODS server [host={}, port={}].", (Object)this.host, (Object)this.port);
            }
        }
        if (connectFailed.get()) {
            throw new IllegalStateException(String.format("Connection error [host=%s, port=%d]", this.host, this.port));
        }
        if (authFailed.get()) {
            throw new IllegalStateException(String.format("Authentication error", new Object[0]));
        }
        IRODSApi.RcComm comm = this.pool.get((int)0).conn.getRcComm();
        String queryString = Versioning.compareVersions(comm.relVersion.substring(4), "4.3.4") > 0 ? "select RESC_MODIFY_TIME, RESC_MODIFY_TIME_MILLIS order by RESC_MODIFY_TIME desc, RESC_MODIFY_TIME_MILLIS desc limit 1" : "select no distinct RESC_MODIFY_TIME, RESC_MODIFY_TIME_MILLIS order by RESC_MODIFY_TIME desc, RESC_MODIFY_TIME_MILLIS desc limit 1";
        List<String> latestRescMtimeRow = IRODSQuery.executeGenQuery2(comm, this.clientUser.getZone(), queryString).get(0);
        String latestRescMtime = String.format("%s.%s", latestRescMtimeRow.get(0), latestRescMtimeRow.get(1));
        List<String> rescCountRow = IRODSQuery.executeGenQuery2(comm, this.clientUser.getZone(), "select count(RESC_ID) limit 1").get(0);
        int rescCount = Integer.parseInt(rescCountRow.get(0));
        this.pool.forEach(ctx -> {
            ctx.latestRescMTime = latestRescMtime;
            ctx.rescCount = rescCount;
        });
    }

    private void refreshConnection(ConnectionContext ctx) {
        if (!this.isConnectionReadyForUse(ctx)) {
            try {
                this.createNewConnection(ctx);
            }
            catch (Exception e) {
                log.error(e.getMessage());
            }
        }
    }

    private boolean isConnectionReadyForUse(ConnectionContext ctx) {
        long elapsed;
        if (!ctx.conn.isConnected()) {
            return false;
        }
        if (this.poolOptions.numberOfRetrievalsBeforeConnectionRefresh.isPresent() && ctx.retrievalCount >= this.poolOptions.numberOfRetrievalsBeforeConnectionRefresh.get()) {
            return false;
        }
        if (this.poolOptions.numberOfSecondsBeforeConnectionRefresh.isPresent() && (elapsed = Instant.now().getEpochSecond() - ctx.ctime) >= (long)this.poolOptions.numberOfSecondsBeforeConnectionRefresh.get().intValue()) {
            return false;
        }
        try {
            if (this.poolOptions.refreshConnectionsWhenResourceChangesDetected) {
                String queryString;
                List<String> latestRescMtimeRow;
                String latestRescMtime;
                boolean inSync = true;
                IRODSApi.RcComm comm = ctx.conn.getRcComm();
                String zone = this.clientUser.getZone();
                List<String> rescCountRow = IRODSQuery.executeGenQuery2(comm, zone, "select count(RESC_ID)").get(0);
                log.debug("Resource count = {}", (Object)rescCountRow.get(0));
                int rescCount = Integer.parseInt(rescCountRow.get(0));
                if (rescCount != ctx.rescCount) {
                    inSync = false;
                    ctx.rescCount = rescCount;
                }
                if (rescCount > 0 && (latestRescMtime = String.format("%s.%s", (latestRescMtimeRow = IRODSQuery.executeGenQuery2(comm, zone, queryString = Versioning.compareVersions(comm.relVersion.substring(4), "4.3.4") > 0 ? "select RESC_MODIFY_TIME, RESC_MODIFY_TIME_MILLIS order by RESC_MODIFY_TIME desc, RESC_MODIFY_TIME_MILLIS desc limit 1" : "select no distinct RESC_MODIFY_TIME, RESC_MODIFY_TIME_MILLIS order by RESC_MODIFY_TIME desc, RESC_MODIFY_TIME_MILLIS desc limit 1").get(0)).get(0), latestRescMtimeRow.get(1))).equals(ctx.latestRescMTime)) {
                    inSync = false;
                    ctx.latestRescMTime = latestRescMtime;
                }
                if (!inSync) {
                    return inSync;
                }
            } else {
                IRODSQuery.executeGenQuery2(ctx.conn.getRcComm(), "select ZONE_NAME where ZONE_TYPE = 'local'");
            }
        }
        catch (Exception e) {
            log.error(e.getMessage());
            return false;
        }
        return true;
    }

    private void createNewConnection(ConnectionContext ctx) throws Exception {
        ctx.conn.disconnect();
        IRODSConnection newConn = new IRODSConnection(this.connOptions);
        newConn.connect(this.host, this.port, this.clientUser);
        if (!this.authenticator.apply(newConn.getRcComm()).booleanValue()) {
            return;
        }
        if (this.poolOptions.numberOfSecondsBeforeConnectionRefresh.isPresent()) {
            ctx.ctime = Instant.now().getEpochSecond();
        }
        if (this.poolOptions.numberOfRetrievalsBeforeConnectionRefresh.isPresent()) {
            ctx.retrievalCount = 0;
        }
        ctx.conn = newConn;
    }

    private static final class ConnectionContext {
        Lock lock = new ReentrantLock();
        AtomicBoolean inUse = new AtomicBoolean();
        IRODSConnection conn;
        long ctime;
        String latestRescMTime;
        int rescCount;
        int retrievalCount;

        private ConnectionContext() {
        }
    }

    public static final class PoolConnection
    implements AutoCloseable {
        private ConnectionContext ctx;

        private PoolConnection(ConnectionContext ctx) {
            this.ctx = ctx;
        }

        public boolean isValid() {
            return this.ctx.conn.isConnected();
        }

        public IRODSApi.RcComm getRcComm() {
            return this.ctx.conn.getRcComm();
        }

        @Override
        public void close() throws Exception {
            this.ctx.inUse.set(false);
        }
    }
}

