/*
 * Copyright © 2019 admin (admin@infrastructurebuilder.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.infrastructurebuilder.nexus.maven;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import org.apache.maven.archiver.MavenArchiver;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.descriptor.PluginDescriptor;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.plugins.assembly.InvalidAssemblerConfigurationException;
import org.apache.maven.plugins.assembly.archive.ArchiveCreationException;
import org.apache.maven.plugins.assembly.archive.AssemblyArchiver;
import org.apache.maven.plugins.assembly.format.AssemblyFormattingException;
import org.apache.maven.plugins.assembly.io.AssemblyReadException;
import org.apache.maven.plugins.assembly.io.AssemblyReader;
import org.apache.maven.plugins.assembly.model.Assembly;
import org.apache.maven.plugins.assembly.mojos.AbstractAssemblyMojo;
import org.apache.maven.plugins.assembly.utils.AssemblyFormatUtils;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.infrastructurebuilder.nexus.AssemblyCreator;
import org.infrastructurebuilder.nexus.Constants;
import org.infrastructurebuilder.nexus.NexusConfigWriter;
import org.infrastructurebuilder.nexus.NexusConfigWriterConfig;
import org.infrastructurebuilder.nexus.NexusConfigWriterConfigSupplier;

/**
 *
 * Not working yet
 *
 * @author mykel.alvis
 *
 */
@Mojo(name = "single", defaultPhase = LifecyclePhase.PACKAGE, requiresProject = true, inheritByDefault = false, requiresDependencyResolution = ResolutionScope.TEST, threadSafe = false)
public class NexusConfigWriterPackagingMojo extends AbstractAssemblyMojo {
  @Parameter(defaultValue = "${mojoExecution}", readonly = true)
  public MojoExecution mojo;

  public NexusConfigWriterPackagingMojo() {
    super();
  }

  /**
   * Controls whether the assembly plugin tries to attach the resulting assembly to the project.
   *
   * @since 2.2-beta-1
   */
  @Parameter(property = "assembly.attach", defaultValue = "true")
  private boolean attach;
  /**
   * Indicates if zip archives (jar,zip etc) being added to the assembly should be compressed again. Compressing again
   * can result in smaller archive size, but gives noticeably longer execution time.
   *
   * @since 2.4
   */
  @Parameter(defaultValue = "true")
  private boolean recompressZippedFiles;

  public boolean isRecompressZippedFiles() {
    return recompressZippedFiles;
  }

  @Parameter(defaultValue = "${plugin}", readonly = true)
  private PluginDescriptor                plugin;
  /**
   * Maven ProjectHelper.
   */
  @Component
  private MavenProjectHelper              projectHelper;

  @Component(hint = Constants.NEXUS)
  private AssemblyReader                  assemblyReader;

  @Component(hint = Constants.NEXUS)
  private NexusConfigWriterConfigSupplier configSupplier;
  @Component
  private AssemblyArchiver                assemblyArchiver;

  @Component
  private AssemblyCreator                 writer;

  @Parameter(required = false, defaultValue = "false")
  private Boolean                         skip = null;

  public void setSkipAssembly(boolean skip) {
    this.skip = skip;
  }

  public void setSkip(boolean skip) {
    this.skip = skip;
  }

  boolean isSkip() {
    return skip != null && skip;
  }

  @Component
  private NexusConfigWriter component;

  void setComponent(NexusConfigWriter component) {
    this.component = component;
  }

  @Parameter(required = false, defaultValue = "${project.build.directory}/generated-resources/nexus")
  public File generatedResourceOutputLocation;

  public void setGeneratedResourceOutputLocation(File generatedResourceOutputLocation) {
    this.generatedResourceOutputLocation = generatedResourceOutputLocation;
  }

  public Path getOutputPath() {
    return Objects.requireNonNull(generatedResourceOutputLocation).toPath().toAbsolutePath();
  }

  @Parameter(required = false, defaultValue = "${project.build.outputDirectory}/assemblies")
  private File assemblyOutputDirectory;

  public void setAssemblyOutputDirectory(File assemblyOutputDirectory) {
    this.assemblyOutputDirectory = assemblyOutputDirectory;
  }

  public Path getAssemblyOutputDirectory() {
    return assemblyOutputDirectory.toPath().toAbsolutePath();
  }

  /**
   * Timestamp for reproducible output archive entries, either formatted as ISO 8601
   * <code>yyyy-MM-dd'T'HH:mm:ssXXX</code> or as an int representing seconds since the epoch (like
   * <a href="https://reproducible-builds.org/docs/source-date-epoch/">SOURCE_DATE_EPOCH</a>).
   *
   * @since 3.2.0
   */
  @Parameter(defaultValue = "${project.build.outputTimestamp}")
  private String                  outputTimestamp;

  @Parameter(required = true, readonly = true, defaultValue = "${project}")
  private MavenProject            project;

  @Parameter(required = true, readonly = true, defaultValue = "${project.build.directory}")
  private File                    projectBuildDirectory;

  @Parameter(required = true, readonly = true, defaultValue = "${project.build.outputDirectory}")
  private File                    projectBuildOutputDirectory;

  @Parameter(required = true)
  private NexusConfigWriterConfig writerConfig;
  /**
   * Specifies the formats of the assembly. Multiple formats can be supplied and the Assembly Plugin will generate an
   * archive for each desired formats. When deploying your project, all file formats specified will also be deployed.
   * A format is specified by supplying one of the following values in a &lt;format&gt; subelement:
   * <ul>
   * <li><em>dir</em> - Creates a directory</li>
   * <li><em>zip</em> - Creates a ZIP file format</li>
   * <li><em>tar</em> - Creates a TAR format</li>
   * <li><em>tar.gz</em> or <em>tgz</em> - Creates a gzip'd TAR format</li>
   * <li><em>tar.bz2</em> or <em>tbz2</em> - Creates a bzip'd TAR format</li>
   * <li><em>tar.snappy</em> - Creates a snappy'd TAR format</li>
   * <li><em>tar.xz</em> or <em>txz</em> - Creates a xz'd TAR format</li>
   * </ul>
   */
  @Parameter
  private List<String>            formats;

  public void setProject(MavenProject project) {
    this.project = Objects.requireNonNull(project);
  }

  public NexusConfigWriterConfig getWriterConfig() {
    return writerConfig;
  }

  public MavenProject getProject() {
    return project;
  }

  Path getProjectBuildDirectory() {
    return Objects.requireNonNull(projectBuildDirectory).toPath().toAbsolutePath();
  }

  Path getProjectBuildOutputDirectory() {
    return projectBuildOutputDirectory.toPath().toAbsolutePath();
  }

  void setProjectBuildDirectory(File projectBuildDirectory) {
    this.projectBuildDirectory = projectBuildDirectory;
  }

  void setProjectBuildOutputDirectory(File projectBuildOutputDirectory) {
    this.projectBuildOutputDirectory = projectBuildOutputDirectory;
  }

  void setup() throws MojoExecutionException, MojoFailureException {
    configSupplier.setT(writerConfig);
    writerConfig.forceOutputPath(getOutputPath());
    writerConfig.forceProjectBuildOutputDirectory(getProjectBuildOutputDirectory());
    writerConfig.forceProjectBuildDirectory(getProjectBuildDirectory());
    writer.setGeneratedOutputDirectory(getOutputPath());
  }

  public void superExecute() throws MojoExecutionException, MojoFailureException {

//        if ( skipAssembly )
//        {
//            getLog().info( "Assemblies have been skipped per configuration of the skipAssembly parameter." );
//            return;
//        }

    // Removed as irrelevant
//        // run only at the execution root.
//        if ( runOnlyAtExecutionRoot && !isThisTheExecutionRoot() )
//        {
//            getLog().info( "Skipping the assembly in this project because it's not the Execution Root" );
//            return;
//        }

    List<Assembly> assemblies;
    try {
      assemblies = assemblyReader.readAssemblies(this);
    } catch (final AssemblyReadException e) {
      throw new MojoExecutionException("Error reading assemblies: " + e.getMessage(), e);
    } catch (final InvalidAssemblerConfigurationException e) {
      throw new MojoFailureException(assemblyReader, e.getMessage(),
          "Mojo configuration is invalid: " + e.getMessage());
    }

    // TODO: include dependencies marked for distribution under certain formats
    // TODO: how, might we plug this into an installer, such as NSIS?

    MavenArchiver mavenArchiver                  = new MavenArchiver();
    Date          outputDate                     = mavenArchiver.parseOutputTimestamp(outputTimestamp);

    boolean       warnedAboutMainProjectArtifact = false;
    for (final Assembly assembly : assemblies) {
      try {
        final String fullName         = AssemblyFormatUtils.getDistributionName(assembly, this);

        List<String> effectiveFormats = formats;
        if (effectiveFormats == null || effectiveFormats.size() == 0) {
          effectiveFormats = assembly.getFormats();
        }
        if (effectiveFormats == null || effectiveFormats.size() == 0) {
          throw new MojoFailureException(
              "No formats specified in the execution parameters or the assembly descriptor.");
        }

        for (final String format : effectiveFormats) {
          final File         destFile = assemblyArchiver.createArchive(assembly, fullName, format, this,
              isRecompressZippedFiles(), getMergeManifestMode(), outputDate);

          final MavenProject project  = getProject();
          final String       type     = project.getArtifact().getType();

          if (attach && destFile.isFile()) {
            if (isAssemblyIdAppended()) {
              projectHelper.attachArtifact(project, format, assembly.getId(), destFile);
            } else if (!"pom".equals(type) && format.equals(type)) {
              if (!warnedAboutMainProjectArtifact) {
                final StringBuilder message = new StringBuilder();

                message.append("Configuration option 'appendAssemblyId' is set to false.");
                message.append("\nInstead of attaching the assembly file: ").append(destFile);
                message.append(", it will become the file for main project artifact.");
                message.append("\nNOTE: If multiple descriptors or descriptor-formats are provided "
                    + "for this project, the value of this file will be " + "non-deterministic!");

                getLog().warn(message);
                warnedAboutMainProjectArtifact = true;
              }

              final File existingFile = project.getArtifact().getFile();
              if ((existingFile != null) && existingFile.exists()) {
                getLog().warn("Replacing pre-existing project main-artifact file: " + existingFile
                    + "\nwith assembly file: " + destFile);
              }

              project.getArtifact().setFile(destFile);
            } else {
              projectHelper.attachArtifact(project, format, null, destFile);
            }
          } else if (attach) {
            getLog().warn("Assembly file: " + destFile + " is not a regular file (it may be a directory). "
                + "It cannot be attached to the project build for installation or " + "deployment.");
          }
        }
      } catch (final ArchiveCreationException | AssemblyFormattingException e) {
        throw new MojoExecutionException("Failed to create assembly: " + e.getMessage(), e);
      } catch (final InvalidAssemblerConfigurationException e) {
        throw new MojoFailureException(assembly, "Assembly is incorrectly configured: " + assembly.getId(),
            "Assembly: " + assembly.getId() + " is not configured correctly: " + e.getMessage());
      }
    }
  }

  @Override
  public void execute() throws MojoExecutionException, MojoFailureException {
    if (skip) {
      getLog().info("Skipping nexus config for " + mojo.getExecutionId());
      return;
    }
    setup();
    try {
      getLog()
          .info("Generating (TF) resources to " + getWriterConfig().getRelativeOutputPath(getProjectBuildDirectory()));
      component.execute(getProjectBuildDirectory(), getWriterConfig());
    } catch (IOException e) {
      throw new MojoExecutionException("Failed to execute", e);
    }
    verifyRemovedParameter("classifier");
    verifyRemovedParameter("descriptor");
    verifyRemovedParameter("descriptorId");
    verifyRemovedParameter("includeSite");

    superExecute();
  }

  private void verifyRemovedParameter(String paramName) {
    Object pluginConfiguration = plugin.getPlugin().getConfiguration();
    if (pluginConfiguration instanceof Xpp3Dom) {
      Xpp3Dom configDom = (Xpp3Dom) pluginConfiguration;

      if (configDom.getChild(paramName) != null) {
        throw new IllegalArgumentException(
            "parameter '" + paramName + "' has been removed from the plugin, please verify documentation.");
      }
    }
  }

}
