/*******************************************************************************
 *  Imixs IX Workflow Technology
 *  Copyright (C) 2001, 2008 Imixs Software Solutions GmbH,  
 *  http://www.imixs.com
 *  
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the terms of the GNU General Public License 
 *  as published by the Free Software Foundation; either version 2 
 *  of the License, or (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 *  General Public License for more details.
 *  
 *  You can receive a copy of the GNU General Public
 *  License at http://www.gnu.org/licenses/gpl.html
 *  
 *  Contributors:  
 *  	Imixs Software Solutions GmbH - initial API and implementation
 *  	Ralph Soika
 *******************************************************************************/

package org.imixs.workflow.jee.ejb;

import java.util.Collection;

import org.imixs.workflow.ItemCollection;

/**
 * The WorkflowService is the JEE Implementation for the IX Workflow API. This
 * interface acts as a service facade and supports basic methods to create,
 * process and access workitems. The Interface extends the core api interface
 * org.imixs.workflow.WorkflowManager with getter methods to fetch collections
 * of workitems. The ModelManager is independent form the IX JEE Entity EJBs and
 * uses the standard IntemCollection Object as a data transfer object to
 * comunitcate with clients.
 * 
 * @author rsoika
 * 
 */
public interface WorkflowService extends org.imixs.workflow.WorkflowManager {

	public static final int SORT_ORDER_CREATED_DESC = 0;
	public static final int SORT_ORDER_CREATED_ASC = 1;
	public static final int SORT_ORDER_MODIFIED_DESC = 2;
	public static final int SORT_ORDER_MODIFIED_ASC = 3;

	/**
	 * Returns a collection of workitems belonging to a specified name. The name
	 * is a username or role contained in the $WriteAccess attribute of the
	 * workitem.
	 * 
	 * 
	 * The method returns only workitems the call has sufficient read access
	 * for.
	 * 
	 * @param name
	 *            = username or role contained in $writeAccess - if null current
	 *            username will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkList(String name, int startpos,
			int count, String type, int sortorder) throws Exception;

	/**
	 * returns a collection of workitems belonging to a specified workflow
	 * group. The behaivor is simmilar to the method getWorkList.
	 * 
	 * @param name
	 *            = workflowgroup the workitems belong to
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByGroup(String name,
			int startpos, int count, String type, int sortorder)
			throws Exception;

	/**
	 * Returns a collection of workitems belonging to a specified $processID
	 * defined by the workflow model. The behaivor is simmilar to the method
	 * getWorkList.
	 * 
	 * @param aID
	 *            = $ProcessID for the workitems to be returned.
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByProcessID(int aID,
			int startpos, int count, String type, int sortorder)
			throws Exception;

	/**
	 * Returns a collection of workitems created by a specified user
	 * (namCreator). The behaivor is simmilar to the method getWorkList.
	 * 
	 * 
	 * @param name
	 *            = username for property namCreator - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByCreator(String name,
			int startpos, int count, String type, int sortorder)
			throws Exception;

	/**
	 * Returns a collection of workitems containing a namOwner property
	 * belonging to a specified username. The namOwner property is typical
	 * controled by the OwnerPlugin
	 * 
	 * @param name
	 *            = username for property namOwner - if null current username
	 *            will be used
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByOwner(String name,
			int startpos, int count, String type, int sortorder)
			throws Exception;

	/**
	 * Returns a collection of workitems where the current user has a
	 * writeAccess. This means the either the username or one of the userroles
	 * is contained in the $writeaccess property
	 * 
	 * 
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * 
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByWriteAccess(int startpos,
			int count, String type, int sortorder) throws Exception;

	/**
	 * Returns a collection of workitems belonging to a specified workitem
	 * identified by the attribute $UniqueIDRef.
	 * 
	 * The behaivor of this Mehtod is simmilar to the method getWorkList.
	 * 
	 * @param aref
	 *            A unique reference to another workitem inside a database *
	 * @param startpos
	 *            = optional start position
	 * @param count
	 *            = optional count - default = -1
	 * @param type
	 *            = defines the type property of the workitems to be returnd.
	 *            can be null
	 * @param sortorder
	 *            = defines sortorder (SORT_ORDER_CREATED_DESC = 0
	 *            SORT_ORDER_CREATED_ASC = 1 SORT_ORDER_MODIFIED_DESC = 2
	 *            SORT_ORDER_MODIFIED_ASC = 3)
	 * @return List of workitems
	 * @throws Exception
	 */
	public Collection<ItemCollection> getWorkListByRef(String aref,
			int startpos, int count, String type, int sortorder)
			throws Exception;

	/**
	 * This method returns an instance of the Imixs JEE EntityService used by
	 * the WorkflowManager Implementation. The method can be used to access the
	 * EntityService during a Plugin call.
	 * 
	 * @return EntityService
	 * @throws Exception
	 */
	public EntityService getEntityService() throws Exception;

	/**
	 * This method returns an instance of the Imixs JEE ModelService used by
	 * the WorkflowManager Implementation. The method can be used to access the
	 * ModelService during a Plugin call.
	 * 
	 * @return EntityService
	 * @throws Exception
	 */
	public ModelService getModelService() throws Exception;

	
	/**
	 * Obtain the java.security.Principal that identifies the caller and returns
	 * the name of this principal.
	 */
	public String getUserName() throws Exception;
	
	/**
	 * Test if the caller has a given security role.
	 * @param rolename
	 * @return true if user is in role
	 */
	public boolean isUserInRole(String rolename) throws Exception;
	
	
	/**
	 * Returns a String array containing all user roles the caller has given  
	 * @return String array with the users access roles
	 * @throws Exception
	 */
	public String[] getUserRoles() throws Exception;
	
}
