package org.icij.datashare.text;

import java.util.HashMap;
import java.util.Map;

import static java.util.Arrays.stream;
import static java.util.Collections.unmodifiableMap;


public class FileExtension {
    private static final Map<String, String> mimeTypesExtensions;

    public static String get(final String mimeType) {
        return mimeTypesExtensions.getOrDefault(mimeType, "bin");
    }

    static {
        mimeTypesExtensions = unmodifiableMap(createMap());
    }

    private static void putList(Map<String, String> map, String extension, String... mimeTypes) {
        stream(mimeTypes).forEach(mt -> map.put(mt, extension));
    }

    private static Map<String, String> createMap() {
        Map<String, String> map = new HashMap<>();
        map.put("x-world/x-3dmf", "3dm");
        map.put("text/vnd.in3d.3dml", "3dml");
        map.put("image/x-3ds", "3ds");
        map.put("video/3gpp2", "3g2");
        map.put("video/3gpp", "3gp");
        map.put("application/x-7z-compressed", "7z");
        map.put("application/x-authorware-bin", "aab");
        map.put("audio/x-aac", "aac");
        map.put("application/x-authorware-map", "aam");
        map.put("application/x-authorware-seg", "aas");
        map.put("text/vnd.abc", "abc");
        map.put("application/x-abiword", "abw");
        map.put("application/pkix-attr-cert", "ac");
        map.put("application/vnd.americandynamics.acc", "acc");
        map.put("application/x-ace-compressed", "ace");
        map.put("text/html", "acgi");
        map.put("application/vnd.acucobol", "acu");
        map.put("application/vnd.acucorp", "acutc");
        map.put("audio/adpcm", "adp");
        map.put("application/vnd.audiograph", "aep");
        map.put("video/animaflex", "afl");
        map.put("application/x-font-type1", "afm");
        map.put("application/vnd.ibm.modcap", "afp");
        map.put("application/vnd.ahead.space", "ahead");
        map.put("application/postscript", "ai");
        map.put("audio/aiff", "aif");
        map.put("application/x-aim", "aim");
        map.put("text/x-audiosoft-intra", "aip");
        map.put("application/vnd.adobe.air-application-installer-package+zip", "air");
        map.put("application/vnd.dvb.ait", "ait");
        map.put("application/vnd.amiga.ami", "ami");
        map.put("application/x-navi-animation", "ani");
        map.put("application/x-nokia-9000-communicator-add-on-software", "aos");
        map.put("application/vnd.android.package-archive", "apk");
        map.put("text/cache-manifest", "appcache");
        map.put("application/x-ms-application", "application");
        map.put("application/vnd.lotus-approach", "apr");
        map.put("application/mime", "aps");
        map.put("application/x-freearc", "arc");
        map.put("application/arj", "arj");
        map.put("image/x-jg", "art");
        map.put("application/pgp-signature", "asc");
        map.put("video/x-ms-asf", "asf");
        map.put("text/x-asm", "asm");
        map.put("application/vnd.accpac.simply.aso", "aso");
        map.put("text/asp", "asp");
        map.put("application/x-mplayer2", "asx");
        map.put("video/x-ms-asf", "asx");
        map.put("video/x-ms-asf-plugin", "asx");
        map.put("application/vnd.acucorp", "atc");
        map.put("application/atom+xml", "atom");
        map.put("application/atomcat+xml", "atomcat");
        map.put("application/atomsvc+xml", "atomsvc");
        map.put("application/vnd.antix.game-component", "atx");
        map.put("audio/basic", "au");
        putList(map, "avi", "application/x-troff-msvideo", "video/avi", "video/msvideo", "video/x-msvideo");
        map.put("video/avs-video", "avs");
        map.put("application/applixware", "aw");
        map.put("application/vnd.airzip.filesecure.azf", "azf");
        map.put("application/vnd.airzip.filesecure.azs", "azs");
        map.put("application/vnd.amazon.ebook", "azw");
        map.put("application/x-msdownload", "bat");
        map.put("application/x-bcpio", "bcpio");
        map.put("application/x-font-bdf", "bdf");
        map.put("application/vnd.syncml.dm+wbxml", "bdm");
        map.put("application/vnd.realvnc.bed", "bed");
        map.put("application/vnd.fujitsu.oasysprs", "bh2");
        putList(map, "bin", "application/mac-binary", "application/macbinary", "application/x-binary", "application/x-macbinary");
        map.put("application/x-blorb", "blb");
        map.put("application/x-blorb", "blorb");
        map.put("image/bmp", "bm");
        map.put("application/vnd.bmi", "bmi");
        putList(map, "bmp", "image/bmp", "image/x-windows-bmp");
        map.put("application/book", "boo");
        map.put("application/vnd.framemaker", "book");
        map.put("application/vnd.previewsystems.box", "box");
        map.put("application/x-bzip2", "boz");
        map.put("application/x-bsh", "bsh");
        map.put("image/prs.btif", "btif");
        map.put("application/x-bzip", "bz");
        map.put("application/x-bzip2", "bz2");
        map.put("text/x-c", "c");
        map.put("text/plain", "c++");
        map.put("application/vnd.cluetrust.cartomobile-config", "c11amc");
        map.put("application/vnd.cluetrust.cartomobile-config-pkg", "c11amz");
        map.put("application/vnd.clonk.c4group", "c4d");
        map.put("application/vnd.clonk.c4group", "c4f");
        map.put("application/vnd.clonk.c4group", "c4g");
        map.put("application/vnd.clonk.c4group", "c4p");
        map.put("application/vnd.clonk.c4group", "c4u");
        map.put("application/vnd.ms-cab-compressed", "cab");
        map.put("audio/x-caf", "caf");
        map.put("application/vnd.tcpdump.pcap", "cap");
        map.put("application/vnd.curl.car", "car");
        map.put("application/vnd.ms-pki.seccat", "cat");
        map.put("application/x-cbr", "cb7");
        map.put("application/x-cbr", "cba");
        map.put("application/x-cbr", "cbr");
        map.put("application/x-cbr", "cbt");
        map.put("application/x-cbr", "cbz");
        putList(map, "cc", "text/plain", "text/x-c");
        map.put("application/clariscad", "ccad");
        map.put("application/x-cocoa", "cco");
        map.put("application/x-director", "cct");
        map.put("application/ccxml+xml", "ccxml");
        map.put("application/vnd.contact.cmsg", "cdbcmsg");
        putList(map, "cdf", "application/cdf", "application/x-cdf", "application/x-netcdf");
        map.put("application/vnd.mediastation.cdkey", "cdkey");
        map.put("application/cdmi-capability", "cdmia");
        map.put("application/cdmi-container", "cdmic");
        map.put("application/cdmi-domain", "cdmid");
        map.put("application/cdmi-object", "cdmio");
        map.put("application/cdmi-queue", "cdmiq");
        map.put("chemical/x-cdx", "cdx");
        map.put("application/vnd.chemdraw+xml", "cdxml");
        map.put("application/vnd.cinderella", "cdy");
        putList(map, "cer", "application/pkix-cert", "application/x-x509-ca-cert");
        map.put("application/x-cfs-compressed", "cfs");
        map.put("image/cgm", "cgm");
        map.put("application/x-chat", "cha");
        map.put("application/x-chat", "chat");
        map.put("application/vnd.ms-htmlhelp", "chm");
        map.put("application/vnd.kde.kchart", "chrt");
        map.put("chemical/x-cif", "cif");
        map.put("application/vnd.anser-web-certificate-issue-initiation", "cii");
        map.put("application/vnd.ms-artgalry", "cil");
        map.put("application/vnd.claymore", "cla");
        putList(map, "class", "application/java", "application/java-byte-code", "application/x-java-class");
        map.put("application/vnd.crick.clicker.keyboard", "clkk");
        map.put("application/vnd.crick.clicker.palette", "clkp");
        map.put("application/vnd.crick.clicker.template", "clkt");
        map.put("application/vnd.crick.clicker.wordbank", "clkw");
        map.put("application/vnd.crick.clicker", "clkx");
        map.put("application/x-msclip", "clp");
        map.put("application/vnd.cosmocaller", "cmc");
        map.put("chemical/x-cmdf", "cmdf");
        map.put("chemical/x-cml", "cml");
        map.put("application/vnd.yellowriver-custom-menu", "cmp");
        map.put("image/x-cmx", "cmx");
        map.put("application/vnd.rim.cod", "cod");
        putList(map, "com", "text/plain");
        map.put("text/plain", "conf");
        map.put("application/x-cpio", "cpio");
        map.put("text/x-c", "cpp");
        putList(map, "cpt", "application/x-compactpro", "application/x-cpt");
        map.put("application/x-mscardfile", "crd");
        putList(map, "crl", "application/pkcs-crl", "application/pkix-crl");
        putList(map, "crt", "application/pkix-cert", "application/x-x509-ca-cert", "application/x-x509-user-cert");
        map.put("application/x-chrome-extension", "crx");
        map.put("application/vnd.rig.cryptonote", "cryptonote");
        putList(map, "csh", "application/x-csh", "text/x-script.csh");
        map.put("chemical/x-csml", "csml");
        map.put("application/vnd.commonspace", "csp");
        putList(map, "css", "application/x-pointplus", "text/css");
        map.put("application/x-director", "cst");
        map.put("text/csv", "csv");
        map.put("application/cu-seeme", "cu");
        map.put("text/vnd.curl", "curl");
        map.put("application/prs.cww", "cww");
        map.put("application/x-director", "cxt");
        map.put("text/x-c", "cxx");
        map.put("model/vnd.collada+xml", "dae");
        map.put("application/vnd.mobius.daf", "daf");
        map.put("application/vnd.dart", "dart");
        map.put("application/vnd.fdsn.seed", "dataless");
        map.put("application/davmount+xml", "davmount");
        map.put("application/docbook+xml", "dbk");
        map.put("application/x-director", "dcr");
        map.put("text/vnd.curl.dcurl", "dcurl");
        map.put("application/vnd.oma.dd2+xml", "dd2");
        map.put("application/vnd.fujixerox.ddd", "ddd");
        map.put("application/x-debian-package", "deb");
        map.put("application/x-deepv", "deepv");
        map.put("text/plain", "def");
        map.put("application/x-x509-ca-cert", "der");
        map.put("application/vnd.dreamfactory", "dfac");
        map.put("application/x-dgc-compressed", "dgc");
        map.put("text/x-c", "dic");
        map.put("video/x-dv", "dif");
        map.put("text/plain", "diff");
        map.put("application/x-director", "dir");
        map.put("application/vnd.mobius.dis", "dis");
        map.put("image/vnd.djvu", "djv");
        map.put("image/vnd.djvu", "djvu");
        putList(map, "dl", "video/dl", "video/x-dl");
        map.put("application/x-msdownload", "dll");
        map.put("application/x-apple-diskimage", "dmg");
        map.put("application/vnd.tcpdump.pcap", "dmp");
        map.put("application/vnd.dna", "dna");
        map.put("application/msword", "doc");
        map.put("application/vnd.ms-word.document.macroenabled.12", "docm");
        map.put("application/vnd.openxmlformats-officedocument.wordprocessingml.document", "docx");
        map.put("application/msword", "dot");
        map.put("application/vnd.ms-word.template.macroenabled.12", "dotm");
        map.put("application/vnd.openxmlformats-officedocument.wordprocessingml.template", "dotx");
        map.put("application/vnd.osgi.dp", "dp");
        map.put("application/vnd.dpgraph", "dpg");
        map.put("audio/vnd.dra", "dra");
        map.put("application/drafting", "drw");
        map.put("text/prs.lines.tag", "dsc");
        map.put("application/dssc+der", "dssc");
        map.put("application/x-dtbook+xml", "dtb");
        map.put("application/xml-dtd", "dtd");
        map.put("audio/vnd.dts", "dts");
        map.put("audio/vnd.dts.hd", "dtshd");
        map.put("video/x-dv", "dv");
        map.put("video/vnd.dvb.file", "dvb");
        map.put("application/x-dvi", "dvi");
        putList(map, "dwf" ,"drawing/x-dwf (old)", "model/vnd.dwf");
        putList(map, "dwg", "application/acad", "image/vnd.dwg", "image/x-dwg");
        map.put("image/vnd.dxf", "dxf");
        map.put("application/vnd.spotfire.dxp", "dxp");
        map.put("application/x-director", "dxr");
        map.put("audio/vnd.nuera.ecelp4800", "ecelp4800");
        map.put("audio/vnd.nuera.ecelp7470", "ecelp7470");
        map.put("audio/vnd.nuera.ecelp9600", "ecelp9600");
        map.put("application/ecmascript", "ecma");
        map.put("application/vnd.novadigm.edm", "edm");
        map.put("application/vnd.novadigm.edx", "edx");
        map.put("application/vnd.picsel", "efif");
        map.put("application/vnd.pg.osasli", "ei6");
        map.put("text/x-script.elisp", "el");
        putList(map, "elc", "application/x-bytecode.elisp (compiled elisp)", "application/x-elc");
        map.put("application/x-msmetafile", "emf");
        map.put("message/rfc822", "eml");
        map.put("application/emma+xml", "emma");
        map.put("application/x-msmetafile", "emz");
        map.put("application/x-envoy", "env");
        map.put("audio/vnd.digital-winds", "eol");
        map.put("application/vnd.ms-fontobject", "eot");
        map.put("application/postscript", "eps");
        map.put("application/epub+zip", "epub");
        map.put("application/x-esrehber", "es");
        map.put("application/vnd.eszigno3+xml", "es3");
        map.put("application/vnd.osgi.subsystem", "esa");
        map.put("application/vnd.epson.esf", "esf");
        map.put("application/vnd.eszigno3+xml", "et3");
        map.put("text/x-setext", "etx");
        map.put("application/x-eva", "eva");
        map.put("text/event-stream", "event-stream");
        putList(map, "evy", "application/envoy", "application/x-envoy");
        map.put("application/x-msdownload", "exe");
        map.put("application/exi", "exi");
        map.put("application/vnd.novadigm.ext", "ext");
        map.put("application/andrew-inset", "ez");
        map.put("application/vnd.ezpix-album", "ez2");
        map.put("application/vnd.ezpix-package", "ez3");
        putList(map, "f", "text/plain", "text/x-fortran");
        map.put("video/x-f4v", "f4v");
        map.put("text/x-fortran", "f77");
        putList(map, "f90", "text/plain", "text/x-fortran");
        map.put("image/vnd.fastbidsheet", "fbs");
        map.put("application/vnd.adobe.formscentral.fcdt", "fcdt");
        map.put("application/vnd.isac.fcs", "fcs");
        map.put("application/vnd.fdf", "fdf");
        putList(map, "fe_launch","application/vnd.denovo.fcselayout-link");
        map.put("application/vnd.fujitsu.oasysgp", "fg5");
        map.put("application/x-director", "fgd");
        map.put("image/x-freehand", "fh");
        map.put("image/x-freehand", "fh4");
        map.put("image/x-freehand", "fh5");
        map.put("image/x-freehand", "fh7");
        map.put("image/x-freehand", "fhc");
        putList(map, "fif", "application/fractals", "image/fif");
        map.put("application/x-xfig", "fig");
        map.put("audio/flac", "flac");
        putList(map, "fli", "video/fli", "video/x-fli");
        map.put("application/vnd.micrografx.flo", "flo");
        map.put("video/x-flv", "flv");
        map.put("application/vnd.kde.kivio", "flw");
        map.put("text/vnd.fmi.flexstor", "flx");
        map.put("text/vnd.fly", "fly");
        map.put("application/vnd.framemaker", "fm");
        map.put("video/x-atomic3d-feature", "fmf");
        map.put("application/vnd.frogans.fnc", "fnc");
        putList(map, "for", "text/plain", "text/x-fortran");
        putList(map, "fpx", "image/vnd.fpx", "image/vnd.net-fpx");
        map.put("application/vnd.framemaker", "frame");
        map.put("application/freeloader", "frl");
        map.put("application/vnd.fsc.weblaunch", "fsc");
        map.put("image/vnd.fst", "fst");
        map.put("application/vnd.fluxtime.clip", "ftc");
        map.put("application/vnd.anser-web-funds-transfer-initiation", "fti");
        map.put("audio/make", "funk");
        map.put("video/vnd.fvt", "fvt");
        map.put("application/vnd.adobe.fxp", "fxp");
        map.put("application/vnd.adobe.fxp", "fxpl");
        map.put("application/vnd.fuzzysheet", "fzs");
        map.put("text/plain", "g");
        map.put("application/vnd.geoplan", "g2w");
        map.put("image/g3fax", "g3");
        map.put("application/vnd.geospace", "g3w");
        map.put("application/vnd.groove-account", "gac");
        map.put("application/x-tads", "gam");
        map.put("application/rpki-ghostbusters", "gbr");
        map.put("application/x-gca-compressed", "gca");
        map.put("model/vnd.gdl", "gdl");
        map.put("application/vnd.dynageo", "geo");
        map.put("application/vnd.geometry-explorer", "gex");
        map.put("application/vnd.geogebra.file", "ggb");
        map.put("application/vnd.geogebra.tool", "ggt");
        map.put("application/vnd.groove-help", "ghf");
        map.put("image/gif", "gif");
        map.put("application/vnd.groove-identity-message", "gim");
        putList(map, "gl", "video/gl", "video/x-gl");
        map.put("application/gml+xml", "gml");
        map.put("application/vnd.gmx", "gmx");
        map.put("application/x-gnumeric", "gnumeric");
        map.put("application/vnd.flographit", "gph");
        map.put("application/gpx+xml", "gpx");
        map.put("application/vnd.grafeq", "gqf");
        map.put("application/vnd.grafeq", "gqs");
        map.put("application/srgs", "gram");
        map.put("application/x-gramps-xml", "gramps");
        map.put("application/vnd.geometry-explorer", "gre");
        map.put("application/vnd.groove-injector", "grv");
        map.put("application/srgs+xml", "grxml");
        map.put("audio/x-gsm", "gsd");
        map.put("application/x-font-ghostscript", "gsf");
        map.put("audio/x-gsm", "gsm");
        map.put("application/x-gsp", "gsp");
        map.put("application/x-gss", "gss");
        map.put("application/x-gtar", "gtar");
        map.put("application/vnd.groove-tool-message", "gtm");
        map.put("model/vnd.gtw", "gtw");
        map.put("text/vnd.graphviz", "gv");
        map.put("application/gxf", "gxf");
        map.put("application/vnd.geonext", "gxt");
        putList(map, "gz", "application/x-compressed", "application/x-gzip");
        putList(map, "gzip", "application/x-gzip", "multipart/x-gzip");
        putList(map, "h", "text/plain", "text/x-h");
        map.put("video/h261", "h261");
        map.put("video/h263", "h263");
        map.put("video/h264", "h264");
        map.put("application/vnd.hal+xml", "hal");
        map.put("application/vnd.hbci", "hbci");
        map.put("application/x-hdf", "hdf");
        map.put("application/x-helpfile", "help");
        map.put("application/vnd.hp-hpgl", "hgl");
        putList(map, "hh", "text/plain", "text/x-h");
        map.put("text/x-script", "hlb");
        putList(map, "hlp", "application/hlp", "application/x-helpfile", "application/x-winhelp");
        map.put("application/vnd.hp-hpgl", "hpg");
        map.put("application/vnd.hp-hpgl", "hpgl");
        map.put("application/vnd.hp-hpid", "hpid");
        map.put("application/vnd.hp-hps", "hps");
        putList(map, "hqx", "application/binhex", "application/binhex4", "application/mac-binhex", "application/mac-binhex40", "application/x-binhex40", "application/x-mac-binhex40");
        map.put("application/hta", "hta");
        map.put("text/x-component", "htc");
        map.put("application/vnd.kenameaapp", "htke");
        map.put("text/html", "htm");
        map.put("text/html", "html");
        map.put("text/html", "htmls");
        map.put("text/webviewhtml", "htt");
        map.put("text/html", "htx");
        map.put("application/vnd.yamaha.hv-dic", "hvd");
        map.put("application/vnd.yamaha.hv-voice", "hvp");
        map.put("application/vnd.yamaha.hv-script", "hvs");
        map.put("application/vnd.intergeo", "i2g");
        map.put("application/vnd.iccprofile", "icc");
        map.put("x-conference/x-cooltalk", "ice");
        map.put("application/vnd.iccprofile", "icm");
        map.put("image/x-icon", "ico");
        map.put("text/calendar", "ics");
        map.put("text/plain", "idc");
        map.put("image/ief", "ief");
        map.put("image/ief", "iefs");
        map.put("text/calendar", "ifb");
        map.put("application/vnd.shana.informed.formdata", "ifm");
        putList(map, "iges", "application/iges", "model/iges");
        map.put("application/vnd.igloader", "igl");
        map.put("application/vnd.insors.igm", "igm");
        putList(map, "igs", "application/iges", "model/iges");
        map.put("application/vnd.micrografx.igx", "igx");
        map.put("application/vnd.shana.informed.interchange", "iif");
        map.put("application/x-ima", "ima");
        map.put("application/x-httpd-imap", "imap");
        map.put("application/vnd.accpac.simply.imp", "imp");
        map.put("application/vnd.ms-ims", "ims");
        map.put("text/plain", "in");
        map.put("application/inf", "inf");
        map.put("application/inkml+xml", "ink");
        map.put("application/inkml+xml", "inkml");
        map.put("application/x-internett-signup", "ins");
        map.put("application/x-install-instructions", "install");
        map.put("application/vnd.astraea-software.iota", "iota");
        map.put("application/x-ip2", "ip");
        map.put("application/ipfix", "ipfix");
        map.put("application/vnd.shana.informed.package", "ipk");
        map.put("application/vnd.ibm.rights-management", "irm");
        map.put("application/vnd.irepository.package+xml", "irp");
        map.put("application/x-iso9660-image", "iso");
        map.put("video/x-isvideo", "isu");
        map.put("audio/it", "it");
        map.put("application/vnd.shana.informed.formtemplate", "itp");
        map.put("application/x-inventor", "iv");
        map.put("application/vnd.immervision-ivp", "ivp");
        map.put("i-world/i-vrml", "ivr");
        map.put("application/vnd.immervision-ivu", "ivu");
        map.put("application/x-livescreen", "ivy");
        map.put("text/vnd.sun.j2me.app-descriptor", "jad");
        map.put("application/vnd.jam", "jam");
        map.put("application/java-archive", "jar");
        putList(map, "jav", "text/plain", "text/x-java-source");
        putList(map, "java", "text/plain", "text/x-java-source");
        map.put("application/x-java-commerce", "jcm");
        putList(map, "jfif", "image/jpeg", "image/pjpeg");
        map.put("image/jpeg", "jfif-tbnl");
        map.put("application/vnd.jisp", "jisp");
        map.put("application/vnd.hp-jlyt", "jlt");
        map.put("application/x-java-jnlp-file", "jnlp");
        map.put("application/vnd.joost.joda-archive", "joda");
        putList(map, "jpe", "image/jpeg", "image/pjpeg");
        putList(map, "jpeg", "image/jpeg", "image/pjpeg");
        putList(map, "jpg", "image/jpeg", "image/pjpeg");
        map.put("video/jpm", "jpgm");
        map.put("video/jpeg", "jpgv");
        map.put("video/jpm", "jpm");
        map.put("image/x-jps", "jps");
        map.put("application/javascript", "js");
        putList(map, "json", "application/json", "text/plain");
        map.put("application/jsonml+json", "jsonml");
        map.put("image/jutvision", "jut");
        putList(map, "kar", "audio/midi", "music/x-karaoke");
        map.put("application/vnd.kde.karbon", "karbon");
        map.put("application/vnd.kde.kformula", "kfo");
        map.put("application/vnd.kidspiration", "kia");
        map.put("application/x-killustrator", "kil");
        map.put("application/vnd.google-earth.kml+xml", "kml");
        map.put("application/vnd.google-earth.kmz", "kmz");
        map.put("application/vnd.kinar", "kne");
        map.put("application/vnd.kinar", "knp");
        map.put("application/vnd.kde.kontour", "kon");
        map.put("application/vnd.kde.kpresenter", "kpr");
        map.put("application/vnd.kde.kpresenter", "kpt");
        map.put("application/vnd.ds-keypoint", "kpxx");
        putList(map, "ksh", "application/x-ksh", "text/x-script.ksh");
        map.put("application/vnd.kde.kspread", "ksp");
        map.put("application/vnd.kahootz", "ktr");
        map.put("image/ktx", "ktx");
        map.put("application/vnd.kahootz", "ktz");
        map.put("application/vnd.kde.kword", "kwd");
        map.put("application/vnd.kde.kword", "kwt");
        putList(map, "la", "audio/nspaudio", "audio/x-nspaudio");
        map.put("audio/x-liveaudio", "lam");
        map.put("application/vnd.las.las+xml", "lasxml");
        map.put("application/x-latex", "latex");
        map.put("application/vnd.llamagraphics.life-balance.desktop", "lbd");
        map.put("application/vnd.llamagraphics.life-balance.exchange+xml", "lbe");
        map.put("application/vnd.hhe.lesson-player", "les");
        putList(map, "lha", "application/lha", "application/x-lha");
        map.put("application/vnd.route66.link66+xml", "link66");
        map.put("text/plain", "list");
        map.put("application/vnd.ibm.modcap", "list3820");
        map.put("application/vnd.ibm.modcap", "listafp");
        putList(map, "lma", "audio/nspaudio", "audio/x-nspaudio");
        map.put("application/x-ms-shortcut", "lnk");
        map.put("text/plain", "log");
        map.put("application/lost+xml", "lostxml");
        map.put("application/vnd.ms-lrm", "lrm");
        putList(map, "lsp", "application/x-lisp", "text/x-script.lisp");
        map.put("text/plain", "lst");
        map.put("text/x-la-asf", "lsx");
        map.put("application/vnd.frogans.ltf", "ltf");
        map.put("application/x-latex", "ltx");
        map.put("text/x-lua", "lua");
        map.put("application/x-lua-bytecode", "luac");
        map.put("audio/vnd.lucent.voice", "lvp");
        map.put("application/vnd.lotus-wordpro", "lwp");
        putList(map, "lzh", "application/x-lzh");
        putList(map, "lzx", "application/lzx", "application/x-lzx");
        putList(map, "m", "text/plain", "text/x-m");
        map.put("application/x-msmediaview", "m13");
        map.put("application/x-msmediaview", "m14");
        map.put("video/mpeg", "m1v");
        map.put("application/mp21", "m21");
        map.put("audio/mpeg", "m2a");
        map.put("video/mpeg", "m2v");
        map.put("audio/mpeg", "m3a");
        map.put("audio/x-mpegurl", "m3u");
        map.put("application/x-mpegURL", "m3u8");
        map.put("audio/mp4", "m4a");
        map.put("application/mp4", "m4p");
        map.put("video/vnd.mpegurl", "m4u");
        map.put("video/x-m4v", "m4v");
        map.put("application/mathematica", "ma");
        map.put("application/mads+xml", "mads");
        map.put("application/vnd.ecowin.chart", "mag");
        map.put("application/vnd.framemaker", "maker");
        map.put("text/troff", "man");
        map.put("text/cache-manifest", "manifest");
        map.put("application/x-navimap", "map");
        map.put("text/x-markdown", "markdown");
        map.put("application/mathml+xml", "mathml");
        map.put("application/mathematica", "mb");
        map.put("application/mbedlet", "mbd");
        map.put("application/vnd.mobius.mbk", "mbk");
        map.put("application/mbox", "mbox");
        map.put("application/x-magic-cap-package-1.0", "mc");
        map.put("application/vnd.medcalcdata", "mc1");
        putList(map, "mcd", "application/mcad", "application/x-mathcad");
        putList(map, "mcf", "image/vasa", "text/mcf");
        map.put("application/netmc", "mcp");
        map.put("text/vnd.curl.mcurl", "mcurl");
        map.put("text/x-markdown", "md");
        map.put("application/x-msaccess", "mdb");
        map.put("image/vnd.ms-modi", "mdi");
        map.put("text/troff", "me");
        map.put("model/mesh", "mesh");
        map.put("application/metalink4+xml", "meta4");
        map.put("application/metalink+xml", "metalink");
        map.put("application/mets+xml", "mets");
        map.put("application/vnd.mfmp", "mfm");
        map.put("application/rpki-manifest", "mft");
        map.put("application/vnd.osgeo.mapguide.package", "mgp");
        map.put("application/vnd.proteus.magazine", "mgz");
        map.put("message/rfc822", "mht");
        map.put("message/rfc822", "mhtml");
        putList(map, "mid", "application/x-midi", "audio/midi", "audio/x-mid", "audio/x-midi", "music/crescendo", "x-music/x-midi");
        putList(map, "midi", "application/x-midi", "audio/midi", "audio/x-mid", "audio/x-midi", "music/crescendo", "x-music/x-midi");
        map.put("application/x-mie", "mie");
        putList(map, "mif", "application/x-frame", "application/x-mif");
        putList(map, "mime", "message/rfc822", "www/mime");
        map.put("video/mj2", "mj2");
        map.put("audio/x-vnd.audioexplosion.mjuicemediafile", "mjf");
        map.put("video/mj2", "mjp2");
        map.put("video/x-motion-jpeg", "mjpg");
        map.put("video/x-matroska", "mk3d");
        map.put("audio/x-matroska", "mka");
        map.put("text/x-markdown", "mkd");
        map.put("video/x-matroska", "mks");
        map.put("video/x-matroska", "mkv");
        map.put("application/vnd.dolby.mlp", "mlp");
        putList(map, "mm", "application/base64", "application/x-meme");
        map.put("application/vnd.chipnuts.karaoke-mmd", "mmd");
        map.put("application/base64", "mme");
        map.put("application/vnd.smaf", "mmf");
        map.put("image/vnd.fujixerox.edmics-mmr", "mmr");
        map.put("video/x-mng", "mng");
        map.put("application/x-msmoney", "mny");
        map.put("application/x-mobipocket-ebook", "mobi");
        putList(map, "mod", "audio/mod", "audio/x-mod");
        map.put("application/mods+xml", "mods");
        map.put("video/quicktime", "moov");
        map.put("video/quicktime", "mov");
        map.put("video/x-sgi-movie", "movie");
        putList(map, "mp2", "audio/mpeg", "audio/x-mpeg", "video/mpeg", "video/x-mpeg", "video/x-mpeq2a");
        map.put("application/mp21", "mp21");
        map.put("audio/mpeg", "mp2a");
        putList(map, "mp3", "audio/mpeg3", "audio/x-mpeg-3", "video/mpeg", "video/x-mpeg");
        map.put("video/mp4", "mp4");
        map.put("audio/mp4", "mp4a");
        map.put("application/mp4", "mp4s");
        map.put("video/mp4", "mp4v");
        putList(map, "mpa", "audio/mpeg", "video/mpeg");
        map.put("application/vnd.mophun.certificate", "mpc");
        map.put("video/mpeg", "mpe");
        map.put("video/mpeg", "mpeg");
        putList(map, "mpg", "audio/mpeg", "video/mpeg");
        map.put("video/mp4", "mpg4");
        map.put("audio/mpeg", "mpga");
        map.put("application/vnd.apple.installer+xml", "mpkg");
        map.put("application/vnd.blueice.multipass", "mpm");
        map.put("application/vnd.mophun.application", "mpn");
        map.put("application/vnd.ms-project", "mpp");
        map.put("application/vnd.ms-project", "mpt");
        map.put("application/x-project", "mpv");
        map.put("application/x-project", "mpx");
        map.put("application/vnd.ibm.minipay", "mpy");
        map.put("application/vnd.mobius.mqy", "mqy");
        map.put("application/marc", "mrc");
        map.put("application/marcxml+xml", "mrcx");
        map.put("text/troff", "ms");
        map.put("application/mediaservercontrol+xml", "mscml");
        map.put("application/vnd.fdsn.mseed", "mseed");
        map.put("application/vnd.mseq", "mseq");
        map.put("application/vnd.epson.msf", "msf");
        map.put("model/mesh", "msh");
        map.put("application/x-msdownload", "msi");
        map.put("application/vnd.mobius.msl", "msl");
        map.put("application/vnd.muvee.style", "msty");
        map.put("model/vnd.mts", "mts");
        map.put("application/vnd.musician", "mus");
        map.put("application/vnd.recordare.musicxml+xml", "musicxml");
        map.put("video/x-sgi-movie", "mv");
        map.put("application/x-msmediaview", "mvb");
        map.put("application/vnd.mfer", "mwf");
        map.put("application/mxf", "mxf");
        map.put("application/vnd.recordare.musicxml", "mxl");
        map.put("application/xv+xml", "mxml");
        map.put("application/vnd.triscape.mxs", "mxs");
        map.put("video/vnd.mpegurl", "mxu");
        map.put("audio/make", "my");
        map.put("application/x-vnd.audioexplosion.mzz", "mzz");
        map.put("application/vnd.nokia.n-gage.symbian.install", "n-gage");
        map.put("text/n3", "n3");
        map.put("image/naplps", "nap");
        map.put("image/naplps", "naplps");
        map.put("application/mathematica", "nb");
        map.put("application/vnd.wolfram.player", "nbp");
        map.put("application/x-netcdf", "nc");
        map.put("application/vnd.nokia.configuration-message", "ncm");
        map.put("application/x-dtbncx+xml", "ncx");
        map.put("text/x-nfo", "nfo");
        map.put("application/vnd.nokia.n-gage.data", "ngdat");
        map.put("image/x-niff", "nif");
        map.put("image/x-niff", "niff");
        map.put("application/vnd.nitf", "nitf");
        map.put("application/x-mix-transfer", "nix");
        map.put("application/vnd.neurolanguage.nlu", "nlu");
        map.put("application/vnd.enliven", "nml");
        map.put("application/vnd.noblenet-directory", "nnd");
        map.put("application/vnd.noblenet-sealer", "nns");
        map.put("application/vnd.noblenet-web", "nnw");
        map.put("image/vnd.net-fpx", "npx");
        map.put("application/x-conference", "nsc");
        map.put("application/vnd.lotus-notes", "nsf");
        map.put("application/vnd.nitf", "ntf");
        map.put("application/x-navidoc", "nvd");
        map.put("message/rfc822", "nws");
        map.put("application/x-nzb", "nzb");
        map.put("application/vnd.fujitsu.oasys2", "oa2");
        map.put("application/vnd.fujitsu.oasys3", "oa3");
        map.put("application/vnd.fujitsu.oasys", "oas");
        map.put("application/x-msbinder", "obd");
        map.put("application/x-tgif", "obj");
        map.put("application/oda", "oda");
        map.put("application/vnd.oasis.opendocument.database", "odb");
        map.put("application/vnd.oasis.opendocument.chart", "odc");
        map.put("application/vnd.oasis.opendocument.formula", "odf");
        map.put("application/vnd.oasis.opendocument.formula-template", "odft");
        map.put("application/vnd.oasis.opendocument.graphics", "odg");
        map.put("application/vnd.oasis.opendocument.image", "odi");
        map.put("application/vnd.oasis.opendocument.text-master", "odm");
        map.put("application/vnd.oasis.opendocument.presentation", "odp");
        map.put("application/vnd.oasis.opendocument.spreadsheet", "ods");
        map.put("application/vnd.oasis.opendocument.text", "odt");
        map.put("audio/ogg", "oga");
        map.put("audio/ogg", "ogg");
        map.put("video/ogg", "ogv");
        map.put("application/ogg", "ogx");
        map.put("application/x-omc", "omc");
        map.put("application/x-omcdatamaker", "omcd");
        map.put("application/x-omcregerator", "omcr");
        map.put("application/omdoc+xml", "omdoc");
        map.put("application/onenote", "onepkg");
        map.put("application/onenote", "onetmp");
        map.put("application/onenote", "onetoc");
        map.put("application/onenote", "onetoc2");
        map.put("application/oebps-package+xml", "opf");
        map.put("text/x-opml", "opml");
        map.put("application/vnd.palm", "oprc");
        map.put("application/vnd.lotus-organizer", "org");
        map.put("application/vnd.yamaha.openscoreformat", "osf");
        map.put("application/vnd.yamaha.openscoreformat.osfpvg+xml", "osfpvg");
        map.put("application/vnd.oasis.opendocument.chart-template", "otc");
        map.put("font/opentype", "otf");
        map.put("application/vnd.oasis.opendocument.graphics-template", "otg");
        map.put("application/vnd.oasis.opendocument.text-web", "oth");
        map.put("application/vnd.oasis.opendocument.image-template", "oti");
        map.put("application/vnd.oasis.opendocument.text-master", "otm");
        map.put("application/vnd.oasis.opendocument.presentation-template", "otp");
        map.put("application/vnd.oasis.opendocument.spreadsheet-template", "ots");
        map.put("application/vnd.oasis.opendocument.text-template", "ott");
        map.put("application/oxps", "oxps");
        map.put("application/vnd.openofficeorg.extension", "oxt");
        map.put("text/x-pascal", "p");
        putList(map, "p10", "application/pkcs10", "application/x-pkcs10");
        putList(map, "p12", "application/pkcs-12", "application/x-pkcs12");
        map.put("application/x-pkcs7-signature", "p7a");
        map.put("application/x-pkcs7-certificates", "p7b");
        putList(map, "p7c", "application/pkcs7-mime", "application/x-pkcs7-mime");
        putList(map, "p7m", "application/pkcs7-mime", "application/x-pkcs7-mime");
        map.put("application/x-pkcs7-certreqresp", "p7r");
        map.put("application/pkcs7-signature", "p7s");
        map.put("application/pkcs8", "p8");
        map.put("application/pro_eng", "part");
        map.put("text/x-pascal", "pas");
        map.put("application/vnd.pawaafile", "paw");
        map.put("application/vnd.powerbuilder6", "pbd");
        map.put("image/x-portable-bitmap", "pbm");
        map.put("application/vnd.tcpdump.pcap", "pcap");
        map.put("application/x-font-pcf", "pcf");
        putList(map, "pcl", "application/vnd.hp-pcl", "application/x-pcl");
        map.put("application/vnd.hp-pclxl", "pclxl");
        map.put("image/x-pict", "pct");
        map.put("application/vnd.curl.pcurl", "pcurl");
        map.put("image/x-pcx", "pcx");
        map.put("application/vnd.palm", "pdb");
        map.put("application/pdf", "pdf");
        map.put("application/x-font-type1", "pfa");
        map.put("application/x-font-type1", "pfb");
        map.put("application/x-font-type1", "pfm");
        map.put("application/font-tdpfr", "pfr");
        map.put("audio/make", "pfunk");
        map.put("application/x-pkcs12", "pfx");
        map.put("image/x-portable-graymap", "pgm");
        map.put("application/x-chess-pgn", "pgn");
        map.put("application/pgp-encrypted", "pgp");
        map.put("text/x-php", "php");
        map.put("image/x-pict", "pic");
        map.put("image/pict", "pict");
        map.put("application/pkixcmp", "pki");
        map.put("application/pkix-pkipath", "pkipath");
        map.put("application/vnd.ms-pki.pko", "pko");
        putList(map, "pl", "text/plain", "text/x-script.perl");
        map.put("application/vnd.3gpp.pic-bw-large", "plb");
        map.put("application/vnd.mobius.plc", "plc");
        map.put("application/vnd.pocketlearn", "plf");
        map.put("application/pls+xml", "pls");
        map.put("application/x-pixclscript", "plx");
        putList(map, "pm", "image/x-xpixmap", "text/x-script.perl-module");
        map.put("application/x-pagemaker", "pm4");
        map.put("application/x-pagemaker", "pm5");
        map.put("application/vnd.ctc-posml", "pml");
        map.put("image/png", "png");
        putList(map, "pnm", "application/x-portable-anymap", "image/x-portable-anymap");
        map.put("application/vnd.macports.portpkg", "portpkg");
        putList(map, "pot", "application/mspowerpoint", "application/vnd.ms-powerpoint");
        map.put("application/vnd.ms-powerpoint.template.macroenabled.12", "potm");
        map.put("application/vnd.openxmlformats-officedocument.presentationml.template", "potx");
        map.put("model/x-pov", "pov");
        map.put("application/vnd.ms-powerpoint", "ppa");
        map.put("application/vnd.ms-powerpoint.addin.macroenabled.12", "ppam");
        map.put("application/vnd.cups-ppd", "ppd");
        map.put("image/x-portable-pixmap", "ppm");
        putList(map, "pps", "application/mspowerpoint", "application/vnd.ms-powerpoint");
        map.put("application/vnd.ms-powerpoint.slideshow.macroenabled.12", "ppsm");
        map.put("application/vnd.openxmlformats-officedocument.presentationml.slideshow", "ppsx");
        putList(map, "ppt", "application/mspowerpoint", "application/powerpoint", "application/vnd.ms-powerpoint", "application/x-mspowerpoint");
        map.put("application/vnd.ms-powerpoint.presentation.macroenabled.12", "pptm");
        map.put("application/vnd.openxmlformats-officedocument.presentationml.presentation", "pptx");
        map.put("application/mspowerpoint", "ppz");
        map.put("application/vnd.palm", "pqa");
        map.put("application/x-mobipocket-ebook", "prc");
        map.put("application/vnd.lotus-freelance", "pre");
        map.put("application/pics-rules", "prf");
        map.put("application/pro_eng", "prt");
        map.put("application/postscript", "ps");
        map.put("application/vnd.3gpp.pic-bw-small", "psb");
        map.put("image/vnd.adobe.photoshop", "psd");
        map.put("application/x-font-linux-psf", "psf");
        map.put("application/pskc+xml", "pskcxml");
        map.put("application/vnd.pvi.ptid1", "ptid");
        map.put("application/x-mspublisher", "pub");
        map.put("application/vnd.3gpp.pic-bw-var", "pvb");
        map.put("paleovu/x-pv", "pvu");
        map.put("application/vnd.3m.post-it-notes", "pwn");
        map.put("application/vnd.ms-powerpoint", "pwz");
        map.put("text/x-script.phyton", "py");
        map.put("audio/vnd.ms-playready.media.pya", "pya");
        map.put("applicaiton/x-bytecode.python", "pyc");
        map.put("application/x-python-code", "pyo");
        map.put("video/vnd.ms-playready.media.pyv", "pyv");
        map.put("application/vnd.epson.quickanime", "qam");
        map.put("application/vnd.intu.qbo", "qbo");
        map.put("audio/vnd.qcelp", "qcp");
        map.put("x-world/x-3dmf", "qd3");
        map.put("x-world/x-3dmf", "qd3d");
        map.put("application/vnd.intu.qfx", "qfx");
        map.put("image/x-quicktime", "qif");
        map.put("application/vnd.publishare-delta-tree", "qps");
        map.put("video/quicktime", "qt");
        map.put("video/x-qtc", "qtc");
        map.put("image/x-quicktime", "qti");
        map.put("image/x-quicktime", "qtif");
        map.put("application/vnd.quark.quarkxpress", "qwd");
        map.put("application/vnd.quark.quarkxpress", "qwt");
        map.put("application/vnd.quark.quarkxpress", "qxb");
        map.put("application/vnd.quark.quarkxpress", "qxd");
        map.put("application/vnd.quark.quarkxpress", "qxl");
        map.put("application/vnd.quark.quarkxpress", "qxt");
        putList(map, "ra", "audio/x-pn-realaudio", "audio/x-pn-realaudio-plugin", "audio/x-realaudio");
        map.put("audio/x-pn-realaudio", "ram");
        map.put("application/x-rar-compressed", "rar");
        putList(map, "ras", "application/x-cmu-raster", "image/cmu-raster", "image/x-cmu-raster");
        map.put("image/cmu-raster", "rast");
        map.put("application/vnd.ipunplugged.rcprofile", "rcprofile");
        map.put("application/rdf+xml", "rdf");
        map.put("application/vnd.data-vision.rdz", "rdz");
        map.put("application/vnd.businessobjects", "rep");
        map.put("application/x-dtbresource+xml", "res");
        map.put("text/x-script.rexx", "rexx");
        map.put("image/vnd.rn-realflash", "rf");
        map.put("image/x-rgb", "rgb");
        map.put("application/reginfo+xml", "rif");
        map.put("audio/vnd.rip", "rip");
        map.put("application/x-research-info-systems", "ris");
        map.put("application/resource-lists+xml", "rl");
        map.put("image/vnd.fujixerox.edmics-rlc", "rlc");
        map.put("application/resource-lists-diff+xml", "rld");
        putList(map, "rm", "application/vnd.rn-realmedia", "audio/x-pn-realaudio");
        map.put("audio/midi", "rmi");
        map.put("audio/x-pn-realaudio", "rmm");
        putList(map, "rmp", "audio/x-pn-realaudio", "audio/x-pn-realaudio-plugin");
        map.put("application/vnd.jcp.javame.midlet-rms", "rms");
        map.put("application/vnd.rn-realmedia-vbr", "rmvb");
        map.put("application/relax-ng-compact-syntax", "rnc");
        putList(map, "rng", "application/ringing-tones", "application/vnd.nokia.ringing-tone");
        map.put("application/vnd.rn-realplayer", "rnx");
        map.put("application/rpki-roa", "roa");
        map.put("text/troff", "roff");
        map.put("image/vnd.rn-realpix", "rp");
        map.put("application/vnd.cloanto.rp9", "rp9");
        map.put("audio/x-pn-realaudio-plugin", "rpm");
        map.put("application/vnd.nokia.radio-presets", "rpss");
        map.put("application/vnd.nokia.radio-preset", "rpst");
        map.put("application/sparql-query", "rq");
        map.put("application/rls-services+xml", "rs");
        map.put("application/rsd+xml", "rsd");
        map.put("application/rss+xml", "rss");
        putList(map, "rt", "text/richtext", "text/vnd.rn-realtext");
        putList(map, "rtf", "application/rtf", "application/x-rtf", "text/richtext");
        putList(map, "rtx", "application/rtf", "text/richtext");
        map.put("video/vnd.rn-realvideo", "rv");
        map.put("text/x-asm", "s");
        map.put("audio/s3m", "s3m");
        map.put("application/vnd.yamaha.smaf-audio", "saf");
        map.put("application/x-tbook", "sbk");
        map.put("application/sbml+xml", "sbml");
        map.put("application/vnd.ibm.secure-container", "sc");
        map.put("application/x-msschedule", "scd");
        putList(map, "scm", "application/x-lotusscreencam", "text/x-script.guile", "text/x-script.scheme", "video/x-scm");
        map.put("application/scvp-cv-request", "scq");
        map.put("application/scvp-cv-response", "scs");
        map.put("text/vnd.curl.scurl", "scurl");
        map.put("application/vnd.stardivision.draw", "sda");
        map.put("application/vnd.stardivision.calc", "sdc");
        map.put("application/vnd.stardivision.impress", "sdd");
        map.put("application/vnd.solent.sdkm+xml", "sdkd");
        map.put("application/vnd.solent.sdkm+xml", "sdkm");
        map.put("text/plain", "sdml");
        putList(map, "sdp", "application/sdp", "application/x-sdp");
        map.put("application/sounder", "sdr");
        map.put("application/vnd.stardivision.writer", "sdw");
        putList(map, "sea", "application/sea", "application/x-sea");
        map.put("application/vnd.seemail", "see");
        map.put("application/vnd.fdsn.seed", "seed");
        map.put("application/vnd.sema", "sema");
        map.put("application/vnd.semd", "semd");
        map.put("application/vnd.semf", "semf");
        map.put("application/java-serialized-object", "ser");
        map.put("application/set", "set");
        map.put("application/set-payment-initiation", "setpay");
        map.put("application/set-registration-initiation", "setreg");
        map.put("application/vnd.hydrostatix.sof-data", "sfd-hdstx");
        map.put("application/vnd.spotfire.sfs", "sfs");
        map.put("text/x-sfv", "sfv");
        map.put("image/sgi", "sgi");
        map.put("application/vnd.stardivision.writer-global", "sgl");
        putList(map, "sgm", "text/sgml", "text/x-sgml");
        putList(map, "sgml", "text/sgml", "text/x-sgml");
        putList(map, "sh", "application/x-bsh", "application/x-sh", "application/x-shar", "text/x-script.sh");
        putList(map, "shar", "application/x-bsh", "application/x-shar");
        map.put("application/shf+xml", "shf");
        putList(map, "shtml", "text/html", "text/x-server-parsed-html");
        map.put("text/vnd.wap.si", "si");
        map.put("application/vnd.wap.sic", "sic");
        map.put("image/x-mrsid-image", "sid");
        map.put("application/pgp-signature", "sig");
        map.put("audio/silk", "sil");
        map.put("model/mesh", "silo");
        map.put("application/vnd.symbian.install", "sis");
        map.put("application/vnd.symbian.install", "sisx");
        putList(map, "sit", "application/x-sit", "application/x-stuffit");
        map.put("application/x-stuffitx", "sitx");
        map.put("application/vnd.koan", "skd");
        map.put("application/vnd.koan", "skm");
        map.put("application/vnd.koan", "skp");
        map.put("application/vnd.koan", "skt");
        map.put("application/x-seelogo", "sl");
        map.put("application/vnd.wap.slc", "slc");
        map.put("application/vnd.ms-powerpoint.slide.macroenabled.12", "sldm");
        map.put("application/vnd.openxmlformats-officedocument.presentationml.slide", "sldx");
        map.put("application/vnd.epson.salt", "slt");
        map.put("application/vnd.stepmania.stepchart", "sm");
        map.put("application/vnd.stardivision.math", "smf");
        map.put("application/smil+xml", "smi");
        map.put("application/smil+xml", "smil");
        map.put("video/x-smv", "smv");
        map.put("application/vnd.stepmania.package", "smzip");
        putList(map, "snd", "audio/basic", "audio/x-adpcm");
        map.put("application/x-font-snf", "snf");
        map.put("application/solids", "sol");
        putList(map, "spc", "application/x-pkcs7-certificates", "text/x-speech");
        map.put("application/vnd.yamaha.smaf-phrase", "spf");
        map.put("application/x-futuresplash", "spl");
        map.put("text/vnd.in3d.spot", "spot");
        map.put("application/scvp-vp-response", "spp");
        map.put("application/scvp-vp-request", "spq");
        map.put("application/x-sprite", "spr");
        map.put("application/x-sprite", "sprite");
        map.put("audio/ogg", "spx");
        map.put("application/x-sql", "sql");
        map.put("application/x-wais-source", "src");
        map.put("application/x-subrip", "srt");
        map.put("application/sru+xml", "sru");
        map.put("application/sparql-results+xml", "srx");
        map.put("application/ssdl+xml", "ssdl");
        map.put("application/vnd.kodak-descriptor", "sse");
        map.put("application/vnd.epson.ssf", "ssf");
        map.put("text/x-server-parsed-html", "ssi");
        map.put("application/streamingmedia", "ssm");
        map.put("application/ssml+xml", "ssml");
        map.put("application/vnd.ms-pki.certstore", "sst");
        map.put("application/vnd.sailingtracker.track", "st");
        map.put("application/vnd.sun.xml.calc.template", "stc");
        map.put("application/vnd.sun.xml.draw.template", "std");
        map.put("application/step", "step");
        map.put("application/vnd.wt.stf", "stf");
        map.put("application/vnd.sun.xml.impress.template", "sti");
        map.put("application/hyperstudio", "stk");
        putList(map, "stl", "application/sla", "application/vnd.ms-pki.stl", "application/x-navistyle");
        map.put("application/step", "stp");
        map.put("application/vnd.pg.format", "str");
        map.put("application/vnd.sun.xml.writer.template", "stw");
        map.put("text/vnd.dvb.subtitle", "sub");
        map.put("application/vnd.sus-calendar", "sus");
        map.put("application/vnd.sus-calendar", "susp");
        map.put("application/x-sv4cpio", "sv4cpio");
        map.put("application/x-sv4crc", "sv4crc");
        map.put("application/vnd.dvb.service", "svc");
        map.put("application/vnd.svd", "svd");
        putList(map, "svf", "image/vnd.dwg", "image/x-dwg");
        map.put("image/svg+xml", "svg");
        map.put("image/svg+xml", "svgz");
        putList(map, "svr", "application/x-world", "x-world/x-svr");
        map.put("application/x-director", "swa");
        map.put("application/x-shockwave-flash", "swf");
        map.put("application/vnd.aristanetworks.swi", "swi");
        map.put("application/vnd.sun.xml.calc", "sxc");
        map.put("application/vnd.sun.xml.draw", "sxd");
        map.put("application/vnd.sun.xml.writer.global", "sxg");
        map.put("application/vnd.sun.xml.impress", "sxi");
        map.put("application/vnd.sun.xml.math", "sxm");
        map.put("application/vnd.sun.xml.writer", "sxw");
        map.put("text/troff", "t");
        map.put("application/x-t3vm-image", "t3");
        map.put("application/vnd.mynfc", "taglet");
        map.put("text/x-speech", "talk");
        map.put("application/vnd.tao.intent-module-archive", "tao");
        map.put("application/x-tar", "tar");
        putList(map, "tbk", "application/toolbook", "application/x-tbook");
        map.put("application/vnd.3gpp2.tcap", "tcap");
        putList(map, "tcl", "application/x-tcl", "text/x-script.tcl");
        map.put("text/x-script.tcsh", "tcsh");
        map.put("application/vnd.smart.teacher", "teacher");
        map.put("application/tei+xml", "tei");
        map.put("application/tei+xml", "teicorpus");
        map.put("application/x-tex", "tex");
        map.put("application/x-texinfo", "texi");
        map.put("application/x-texinfo", "texinfo");
        putList(map, "text", "application/plain", "text/plain");
        map.put("application/thraud+xml", "tfi");
        map.put("application/x-tex-tfm", "tfm");
        map.put("image/x-tga", "tga");
        putList(map, "tgz", "application/gnutar", "application/x-compressed");
        map.put("application/vnd.ms-officetheme", "thmx");
        putList(map, "tif", "image/tiff", "image/x-tiff");
        putList(map, "tiff", "image/tiff", "image/x-tiff");
        map.put("application/vnd.tmobile-livetv", "tmo");
        map.put("application/x-bittorrent", "torrent");
        map.put("application/vnd.groove-tool-template", "tpl");
        map.put("application/vnd.trid.tpt", "tpt");
        map.put("text/troff", "tr");
        map.put("application/vnd.trueapp", "tra");
        map.put("application/x-msterminal", "trm");
        map.put("video/MP2T", "ts");
        map.put("application/timestamped-data", "tsd");
        map.put("audio/tsp-audio", "tsi");
        putList(map, "tsp", "application/dsptype", "audio/tsplayer");
        map.put("text/tab-separated-values", "tsv");
        map.put("application/x-font-ttf", "ttc");
        map.put("application/x-font-ttf", "ttf");
        map.put("text/turtle", "ttl");
        map.put("image/florian", "turbot");
        map.put("application/vnd.simtech-mindmapper", "twd");
        map.put("application/vnd.simtech-mindmapper", "twds");
        map.put("application/vnd.genomatix.tuxedo", "txd");
        map.put("application/vnd.mobius.txf", "txf");
        map.put("text/plain", "txt");
        map.put("application/x-authorware-bin", "u32");
        map.put("application/x-debian-package", "udeb");
        map.put("application/vnd.ufdl", "ufd");
        map.put("application/vnd.ufdl", "ufdl");
        map.put("text/x-uil", "uil");
        map.put("application/x-glulx", "ulx");
        map.put("application/vnd.umajin", "umj");
        map.put("text/uri-list", "uni");
        map.put("text/uri-list", "unis");
        map.put("application/vnd.unity", "unityweb");
        map.put("application/i-deas", "unv");
        map.put("application/vnd.uoml+xml", "uoml");
        map.put("text/uri-list", "uri");
        map.put("text/uri-list", "uris");
        map.put("text/uri-list", "urls");
        putList(map, "ustar", "application/x-ustar", "multipart/x-ustar");
        map.put("application/vnd.uiq.theme", "utz");
        putList(map, "uu", "text/x-uuencode");
        map.put("text/x-uuencode", "uue");
        map.put("audio/vnd.dece.audio", "uva");
        map.put("application/vnd.dece.data", "uvd");
        map.put("application/vnd.dece.data", "uvf");
        map.put("image/vnd.dece.graphic", "uvg");
        map.put("video/vnd.dece.hd", "uvh");
        map.put("image/vnd.dece.graphic", "uvi");
        map.put("video/vnd.dece.mobile", "uvm");
        map.put("video/vnd.dece.pd", "uvp");
        map.put("video/vnd.dece.sd", "uvs");
        map.put("application/vnd.dece.ttml+xml", "uvt");
        map.put("video/vnd.uvvu.mp4", "uvu");
        map.put("video/vnd.dece.video", "uvv");
        map.put("audio/vnd.dece.audio", "uvva");
        map.put("application/vnd.dece.data", "uvvd");
        map.put("application/vnd.dece.data", "uvvf");
        map.put("image/vnd.dece.graphic", "uvvg");
        map.put("video/vnd.dece.hd", "uvvh");
        map.put("image/vnd.dece.graphic", "uvvi");
        map.put("video/vnd.dece.mobile", "uvvm");
        map.put("video/vnd.dece.pd", "uvvp");
        map.put("video/vnd.dece.sd", "uvvs");
        map.put("application/vnd.dece.ttml+xml", "uvvt");
        map.put("video/vnd.uvvu.mp4", "uvvu");
        map.put("video/vnd.dece.video", "uvvv");
        map.put("application/vnd.dece.unspecified", "uvvx");
        map.put("application/vnd.dece.zip", "uvvz");
        map.put("application/vnd.dece.unspecified", "uvx");
        map.put("application/vnd.dece.zip", "uvz");
        map.put("text/vcard", "vcard");
        map.put("application/x-cdlink", "vcd");
        map.put("text/x-vcard", "vcf");
        map.put("application/vnd.groove-vcard", "vcg");
        map.put("text/x-vcalendar", "vcs");
        map.put("application/vnd.vcx", "vcx");
        map.put("application/vda", "vda");
        map.put("video/vdo", "vdo");
        map.put("application/groupwise", "vew");
        map.put("application/vnd.visionary", "vis");
        putList(map, "viv", "video/vivo", "video/vnd.vivo");
        putList(map, "vivo", "video/vivo", "video/vnd.vivo");
        map.put("application/vocaltec-media-desc", "vmd");
        map.put("application/vocaltec-media-file", "vmf");
        map.put("video/x-ms-vob", "vob");
        putList(map, "voc", "audio/voc", "audio/x-voc");
        map.put("application/vnd.stardivision.writer", "vor");
        map.put("video/vosaic", "vos");
        map.put("application/x-authorware-bin", "vox");
        map.put("audio/x-twinvq-plugin", "vqe");
        map.put("audio/x-twinvq", "vqf");
        map.put("audio/x-twinvq-plugin", "vql");
        putList(map, "vrml", "application/x-vrml", "model/vrml", "x-world/x-vrml");
        map.put("x-world/x-vrt", "vrt");
        map.put("application/vnd.visio", "vsd");
        map.put("application/vnd.vsf", "vsf");
        map.put("application/vnd.visio", "vss");
        map.put("application/vnd.visio", "vst");
        map.put("application/vnd.visio", "vsw");
        map.put("text/vtt", "vtt");
        map.put("model/vnd.vtu", "vtu");
        map.put("application/voicexml+xml", "vxml");
        map.put("application/x-director", "w3d");
        map.put("application/wordperfect6.0", "w60");
        map.put("application/wordperfect6.1", "w61");
        map.put("application/msword", "w6w");
        map.put("application/x-doom", "wad");
        putList(map, "wav", "audio/wav", "audio/x-wav");
        map.put("audio/x-ms-wax", "wax");
        map.put("application/x-qpro", "wb1");
        map.put("image/vnd.wap.wbmp", "wbmp");
        map.put("application/vnd.criticaltools.wbs+xml", "wbs");
        map.put("application/vnd.wap.wbxml", "wbxml");
        map.put("application/vnd.ms-works", "wcm");
        map.put("application/vnd.ms-works", "wdb");
        map.put("image/vnd.ms-photo", "wdp");
        map.put("application/vnd.xara", "web");
        map.put("audio/webm", "weba");
        map.put("application/x-web-app-manifest+json", "webapp");
        map.put("video/webm", "webm");
        map.put("image/webp", "webp");
        map.put("application/vnd.pmi.widget", "wg");
        map.put("application/widget", "wgt");
        map.put("application/msword", "wiz");
        map.put("application/x-123", "wk1");
        map.put("application/vnd.ms-works", "wks");
        map.put("video/x-ms-wm", "wm");
        map.put("audio/x-ms-wma", "wma");
        map.put("application/x-ms-wmd", "wmd");
        map.put("application/x-msmetafile", "wmf");
        map.put("text/vnd.wap.wml", "wml");
        map.put("application/vnd.wap.wmlc", "wmlc");
        map.put("text/vnd.wap.wmlscript", "wmls");
        map.put("application/vnd.wap.wmlscriptc", "wmlsc");
        map.put("video/x-ms-wmv", "wmv");
        map.put("video/x-ms-wmx", "wmx");
        map.put("application/x-msmetafile", "wmz");
        map.put("application/x-font-woff", "woff");
        map.put("application/msword", "word");
        map.put("application/wordperfect", "wp");
        putList(map, "wp5", "application/wordperfect", "application/wordperfect6.0");
        map.put("application/wordperfect", "wp6");
        putList(map, "wpd", "application/wordperfect", "application/x-wpwin");
        map.put("application/vnd.ms-wpl", "wpl");
        map.put("application/vnd.ms-works", "wps");
        map.put("application/x-lotus", "wq1");
        map.put("application/vnd.wqd", "wqd");
        putList(map, "wri", "application/mswrite", "application/x-wri");
        putList(map, "wrl", "application/x-world", "model/vrml", "x-world/x-vrml");
        putList(map, "wrz", "model/vrml", "x-world/x-vrml");
        map.put("text/scriplet", "wsc");
        map.put("application/wsdl+xml", "wsdl");
        map.put("application/wspolicy+xml", "wspolicy");
        map.put("application/x-wais-source", "wsrc");
        map.put("application/vnd.webturbo", "wtb");
        map.put("application/x-wintalk", "wtk");
        map.put("video/x-ms-wvx", "wvx");
        map.put("image/png", "x-png");
        map.put("application/x-authorware-bin", "x32");
        map.put("model/x3d+xml", "x3d");
        map.put("model/x3d+binary", "x3db");
        map.put("model/x3d+binary", "x3dbz");
        map.put("model/x3d+vrml", "x3dv");
        map.put("model/x3d+vrml", "x3dvz");
        map.put("model/x3d+xml", "x3dz");
        map.put("application/xaml+xml", "xaml");
        map.put("application/x-silverlight-app", "xap");
        map.put("application/vnd.xara", "xar");
        map.put("application/x-ms-xbap", "xbap");
        map.put("application/vnd.fujixerox.docuworks.binder", "xbd");
        putList(map, "xbm", "image/x-xbitmap", "image/x-xbm", "image/xbm");
        map.put("application/xcap-diff+xml", "xdf");
        map.put("application/vnd.syncml.dm+xml", "xdm");
        map.put("application/vnd.adobe.xdp+xml", "xdp");
        map.put("video/x-amt-demorun", "xdr");
        map.put("application/dssc+xml", "xdssc");
        map.put("application/vnd.fujixerox.docuworks", "xdw");
        map.put("application/xenc+xml", "xenc");
        map.put("application/patch-ops-error+xml", "xer");
        map.put("application/vnd.adobe.xfdf", "xfdf");
        map.put("application/vnd.xfdl", "xfdl");
        map.put("xgl/drawing", "xgz");
        map.put("application/xhtml+xml", "xht");
        map.put("application/xhtml+xml", "xhtml");
        map.put("application/xv+xml", "xhvml");
        map.put("image/vnd.xiff", "xif");
        map.put("application/excel", "xl");
        putList(map, "xla", "application/excel", "application/x-excel", "application/x-msexcel");
        map.put("application/vnd.ms-excel.addin.macroenabled.12", "xlam");
        putList(map, "xlb", "application/excel", "application/vnd.ms-excel", "application/x-excel");
        putList(map, "xlc", "application/excel", "application/vnd.ms-excel", "application/x-excel");
        putList(map, "xld", "application/excel", "application/x-excel");
        map.put("application/x-xliff+xml", "xlf");
        putList(map, "xlk", "application/excel", "application/x-excel");
        putList(map, "xll", "application/excel", "application/vnd.ms-excel", "application/x-excel");
        putList(map, "xlm", "application/excel", "application/vnd.ms-excel", "application/x-excel");
        putList(map, "xls", "application/excel", "application/vnd.ms-excel", "application/x-excel", "application/x-msexcel");
        map.put("application/vnd.ms-excel.sheet.binary.macroenabled.12", "xlsb");
        map.put("application/vnd.ms-excel.sheet.macroenabled.12", "xlsm");
        map.put("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", "xlsx");
        putList(map, "xlt", "application/excel", "application/x-excel");
        map.put("application/vnd.ms-excel.template.macroenabled.12", "xltm");
        map.put("application/vnd.openxmlformats-officedocument.spreadsheetml.template", "xltx");
        putList(map, "xlv", "application/excel", "application/x-excel");
        putList(map, "xlw", "application/excel", "application/vnd.ms-excel", "application/x-excel", "application/x-msexcel");
        map.put("audio/xm", "xm");
        putList(map, "xml", "application/xml", "text/xml");
        map.put("xgl/movie", "xmz");
        map.put("application/vnd.olpc-sugar", "xo");
        map.put("application/xop+xml", "xop");
        map.put("application/xml", "xpdl");
        map.put("application/x-xpinstall", "xpi");
        map.put("application/x-vnd.ls-xpix", "xpix");
        map.put("application/xproc+xml", "xpl");
        putList(map, "xpm", "image/x-xpixmap", "image/xpm");
        map.put("application/vnd.is-xpr", "xpr");
        map.put("application/vnd.ms-xpsdocument", "xps");
        map.put("application/vnd.intercon.formnet", "xpw");
        map.put("application/vnd.intercon.formnet", "xpx");
        map.put("application/xml", "xsl");
        map.put("application/xslt+xml", "xslt");
        map.put("application/vnd.syncml+xml", "xsm");
        map.put("application/xspf+xml", "xspf");
        map.put("video/x-amt-showrun", "xsr");
        map.put("application/vnd.mozilla.xul+xml", "xul");
        map.put("application/xv+xml", "xvm");
        map.put("application/xv+xml", "xvml");
        putList(map, "xwd", "image/x-xwd", "image/x-xwindowdump");
        map.put("chemical/x-xyz", "xyz");
        map.put("application/x-xz", "xz");
        map.put("application/yang", "yang");
        map.put("application/yin+xml", "yin");
        putList(map, "z", "application/x-compress", "application/x-compressed");
        map.put("application/x-zmachine", "z1");
        map.put("application/x-zmachine", "z2");
        map.put("application/x-zmachine", "z3");
        map.put("application/x-zmachine", "z4");
        map.put("application/x-zmachine", "z5");
        map.put("application/x-zmachine", "z6");
        map.put("application/x-zmachine", "z7");
        map.put("application/x-zmachine", "z8");
        map.put("application/vnd.zzazz.deck+xml", "zaz");
        putList(map, "zip", "application/x-compressed", "application/x-zip-compressed", "application/zip", "multipart/x-zip");
        map.put("application/vnd.zul", "zir");
        map.put("application/vnd.zul", "zirz");
        map.put("application/vnd.handheld-entertainment+xml", "zmm");
        map.put("text/x-script.zsh", "zsh");
        map.put("application/vnd.lotus-1-2-3", "123");
        return map;
    }
}
