package org.huiche.apt;

import com.querydsl.core.types.PathMetadataFactory;
import com.querydsl.core.types.dsl.BooleanPath;
import com.querydsl.core.types.dsl.DatePath;
import com.querydsl.core.types.dsl.DateTimePath;
import com.querydsl.core.types.dsl.EnumPath;
import com.querydsl.core.types.dsl.NumberPath;
import com.querydsl.core.types.dsl.SimplePath;
import com.querydsl.core.types.dsl.StringPath;
import com.querydsl.core.types.dsl.TimePath;
import com.querydsl.sql.ColumnMetadata;
import com.querydsl.sql.PrimaryKey;
import com.querydsl.sql.RelationalPathBase;
import com.querydsl.sql.types.EnumByNameType;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.huiche.annotation.Column;
import org.huiche.annotation.Table;
import org.huiche.dao.support.EnumTypePool;
import org.huiche.support.NamingUtil;
import org.huiche.support.TypeMapping;

import javax.annotation.processing.Messager;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.Elements;
import javax.lang.model.util.Types;
import javax.tools.Diagnostic;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.JDBCType;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * @author Maning
 */
public class QuerydslMapperGenerator {
    private final Elements elementUtils;
    private final Messager messager;
    private final Types types;


    public QuerydslMapperGenerator(Elements elementUtils, Messager messager, Types types) {
        this.elementUtils = elementUtils;
        this.messager = messager;
        this.types = types;
    }

    public TypeSpec createMapper(TypeElement entity, String packageName) {
        messager.printMessage(Diagnostic.Kind.NOTE, "create mapper for:" + entity.getQualifiedName());
        String entityName = entity.getSimpleName().toString();
        ClassName entityClass = ClassName.get(entity);
        Table table = entity.getAnnotation(Table.class);
        List<FieldSpec> fields = new ArrayList<>();
        List<String> pks = new ArrayList<>();
        List<CodeBlock> metaList = new ArrayList<>();
        List<CodeBlock> enumList = new ArrayList<>();
        int index = 1;
        for (VariableElement field : scanFields(entity)) {
            Set<Modifier> mods = field.getModifiers();
            if (mods.contains(Modifier.STATIC) || mods.contains(Modifier.TRANSIENT)) {
                continue;
            }
            Column column = field.getAnnotation(Column.class);
            if (column != null && !org.huiche.support.PrimaryKey.NOT_PK.equals(column.primaryKey())) {
                pks.add(field.getSimpleName().toString());
            }
            fields.add(createField(field, column, index++, metaList, enumList));
        }
        ClassName mapperClass = ClassName.get(packageName, "Q" + entityName);
        String mapperVar = NamingUtil.pascal2camel(entityName);
        TypeSpec.Builder builder = TypeSpec.classBuilder(mapperClass)
                .addModifiers(Modifier.PUBLIC)
                .addJavadoc("generated by huiche-apt")
                .superclass(ParameterizedTypeName.get(ClassName.get(RelationalPathBase.class), entityClass))
                .addField(FieldSpec.builder(mapperClass, NamingUtil.camel2snake(entityName).toUpperCase(), Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
                        .initializer("new $T($S)", mapperClass, NamingUtil.camel2snake(mapperVar)).build())
                .addFields(fields);
        if (!pks.isEmpty()) {
            builder.addField(createPrimaryKey(entityClass, pks));
        }
        builder.addMethod(createConstructor(entityClass, table));
        builder.addMethod(createMetaMethod(metaList));
        if (!enumList.isEmpty()) {
            CodeBlock.Builder staticBuilder = CodeBlock.builder();
            for (CodeBlock code : enumList) {
                staticBuilder.addStatement(code);
            }
            builder.addStaticBlock(staticBuilder.build());
        }
        return builder.build();
    }

    private List<VariableElement> scanFields(TypeElement entity) {
        List<VariableElement> list = new ArrayList<>();
        TypeMirror father = entity.getSuperclass();
        if (father != null && !Object.class.getCanonicalName().equals(getCanonicalName(father))) {
            Element fatherEle = types.asElement(father);
            if (fatherEle != null && fatherEle.getKind() == ElementKind.CLASS) {
                list.addAll(scanFields((TypeElement) types.asElement(father)));
            }
        }
        list.addAll(ElementFilter.fieldsIn(elementUtils.getAllMembers(entity)));
        return list;
    }

    private MethodSpec createConstructor(ClassName clazz, Table table) {
        return MethodSpec.constructorBuilder()
                .addModifiers(Modifier.PUBLIC)
                .addParameter(String.class, "variable")
                .addStatement("super($T.class, $T.forVariable(variable), $S, $S)",
                        clazz, PathMetadataFactory.class, "".equals(table.schema()) ? "null" : table.schema(),
                        "".equals(table.name()) ? NamingUtil.camel2snake(clazz.simpleName()) : table.name())
                .addStatement("addMetadata()").build();
    }

    private MethodSpec createMetaMethod(List<CodeBlock> metaList) {
        MethodSpec.Builder builder = MethodSpec.methodBuilder("addMetadata").addModifiers(Modifier.PUBLIC).returns(TypeName.VOID);
        metaList.forEach(builder::addStatement);
        return builder.build();
    }


    private FieldSpec createField(VariableElement element, Column column, int index, List<CodeBlock> metaList, List<CodeBlock> enumList) {
        TypeMirror type = element.asType();
        String typeClassName = getCanonicalName(type);
        boolean isEnum = false;
        String javaName = element.getSimpleName().toString();
        String columnName = NamingUtil.camel2snake(javaName);
        JDBCType jdbcType = TypeMapping.parseJdbcType(typeClassName);
        boolean isPk = false;
        boolean nullable = true;
        Integer length = null;
        Integer precision = null;
        if (column != null) {
            columnName = "".equals(column.name()) ? columnName : column.name();
            jdbcType = JDBCType.NULL.equals(column.jdbcType()) ? jdbcType : column.jdbcType();
            isPk = !org.huiche.support.PrimaryKey.NOT_PK.equals(column.primaryKey());
            nullable = column.nullable();
            length = column.length() >= 0 ? column.length() : null;
            precision = column.precision() >= 0 ? column.precision() : null;
        }
        if (jdbcType == null) {
            jdbcType = TypeMapping.parseJdbcType(typeClassName);
        }
        if (jdbcType == null && type.getKind() == TypeKind.DECLARED) {
            Element enumEle = types.asElement(type);
            if (enumEle.getKind() == ElementKind.ENUM) {
                jdbcType = TypeMapping.parseJdbcType(Enum.class.getCanonicalName());
                isEnum = true;
            }
        }
        if (jdbcType == null) {
            throw new RuntimeException("can not support jdbcType in:" + javaName);
        }

        TypeName typeName = TypeName.get(type);
        FieldSpec spec = null;
        if (Objects.equals(Boolean.class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(BooleanPath.class, javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createBoolean($S)", javaName).build();
        } else if (Objects.equals(Byte.class.getCanonicalName(), typeClassName) || Objects.equals(Short.class.getCanonicalName(), typeClassName) || Objects.equals(Integer.class.getCanonicalName(), typeClassName) || Objects.equals(Long.class.getCanonicalName(), typeClassName) || Objects.equals(Float.class.getCanonicalName(), typeClassName) || Objects.equals(Double.class.getCanonicalName(), typeClassName) || Objects.equals(BigInteger.class.getCanonicalName(), typeClassName) || Objects.equals(BigDecimal.class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(NumberPath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createNumber($S, $T.class)", javaName, typeName).build();
        } else if (Objects.equals(String.class.getCanonicalName(), typeClassName)) {
            if (Objects.equals(String.class.getCanonicalName(), typeClassName) && length == null) {
                length = 255;
            }
            spec = FieldSpec.builder(StringPath.class, javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createString($S)", javaName).build();
        } else if (Objects.equals(Character.class.getCanonicalName(), typeClassName)) {
            length = 1;
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(SimplePath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createSimple($S, $T.class)", javaName, typeName).build();
        } else if (Objects.equals(LocalTime.class.getCanonicalName(), typeClassName) || Objects.equals(OffsetTime.class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(TimePath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createTime($S, $T.class)", javaName, typeName).build();
        } else if (Objects.equals(LocalDate.class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(DatePath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createDate($S, $T.class)", javaName, typeName).build();
        } else if (Objects.equals(Date.class.getCanonicalName(), typeClassName) || Objects.equals(LocalDateTime.class.getCanonicalName(), typeClassName) || Objects.equals(OffsetDateTime.class.getCanonicalName(), typeClassName) || Objects.equals(ZonedDateTime.class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(DateTimePath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createDateTime($S, $T.class)", javaName, typeName).build();
        } else if (Objects.equals(byte[].class.getCanonicalName(), typeClassName)) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(SimplePath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createSimple($S, $T.class)", javaName, typeName).build();
        } else if (isEnum) {
            spec = FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(EnumPath.class), typeName), javaName, Modifier.PUBLIC, Modifier.FINAL).initializer("createEnum($S, $T.class)", javaName, typeName).build();
            enumList.add(CodeBlock.of("$T.put($T.class, new $T<>($T.class))", EnumTypePool.class, type, EnumByNameType.class, type));
        }
        if (spec == null) {
            throw new RuntimeException("can not support spec in:" + javaName);
        }
        String format = "addMetadata($L, $T.named($S).withIndex($L).ofType($T.$L)";
        if (isPk || !nullable) {
            format += ".notNull()";
        }
        if (length != null) {
            format += ".withSize(" + length + ")";
            if (precision != null) {
                format += ".withDigits(" + precision + ")";
            }
        }
        format += ")";
        metaList.add(CodeBlock.of(format, javaName, ColumnMetadata.class, columnName, index, java.sql.Types.class, jdbcType.getName()));
        return spec;
    }

    private FieldSpec createPrimaryKey(ClassName entityClass, List<String> pks) {
        return FieldSpec.builder(ParameterizedTypeName.get(ClassName.get(PrimaryKey.class), entityClass), "primary", Modifier.PUBLIC, Modifier.FINAL).initializer("createPrimaryKey(" + String.join(", ", pks) + ")").build();
    }

    private String getCanonicalName(TypeMirror typeMirror) {
        return TypeName.get(typeMirror).toString();
    }
}
