/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import org.apache.commons.lang3.SystemUtils;

public class FileUtils {
    private static final int WINDOWS_RETRY_COUNT = 5;

    public static void deleteRecursively(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        Path path = directory.toPath();
        FileUtils.deletePathRecursively(path);
    }

    public static void deletePathRecursively(Path path) throws IOException {
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                FileUtils.deleteFileWithRetries(file, 0);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static boolean deleteFile(File file) {
        boolean deleted;
        if (!file.exists()) {
            return true;
        }
        int count = 0;
        do {
            if (deleted = file.delete()) continue;
            ++count;
            FileUtils.waitAndThenTriggerGC();
        } while (!deleted && count <= 5);
        return deleted;
    }

    public static void moveFile(File toMove, File target) throws IOException {
        if (!toMove.exists()) {
            throw new FileNotFoundException("Source file[" + toMove.getAbsolutePath() + "] not found");
        }
        if (target.exists()) {
            throw new IOException("Target file[" + target.getAbsolutePath() + "] already exists");
        }
        if (toMove.renameTo(target)) {
            return;
        }
        if (toMove.isDirectory()) {
            Files.createDirectories(target.toPath(), new FileAttribute[0]);
            FileUtils.copyRecursively(toMove, target);
            FileUtils.deleteRecursively(toMove);
        } else {
            FileUtils.copyFile(toMove, target);
            FileUtils.deleteFile(toMove);
        }
    }

    public static File moveFileToDirectory(File toMove, File targetDirectory) throws IOException {
        if (!targetDirectory.isDirectory()) {
            throw new IllegalArgumentException("Move target must be a directory, not " + targetDirectory);
        }
        File target = new File(targetDirectory, toMove.getName());
        FileUtils.moveFile(toMove, target);
        return target;
    }

    public static boolean renameFile(File srcFile, File renameToFile) throws IOException {
        boolean renamed;
        if (!srcFile.exists()) {
            throw new FileNotFoundException("Source file[" + srcFile.getName() + "] not found");
        }
        if (renameToFile.exists()) {
            throw new FileNotFoundException("Target file[" + renameToFile.getName() + "] already exists");
        }
        if (!renameToFile.getParentFile().isDirectory()) {
            throw new FileNotFoundException("Target directory[" + renameToFile.getParent() + "] does not exists");
        }
        int count = 0;
        do {
            if (renamed = srcFile.renameTo(renameToFile)) continue;
            ++count;
            FileUtils.waitAndThenTriggerGC();
        } while (!renamed && count <= 5);
        return renamed;
    }

    public static void truncateFile(SeekableByteChannel fileChannel, long position) throws IOException {
        int count = 0;
        boolean success = false;
        IOException cause = null;
        do {
            ++count;
            try {
                fileChannel.truncate(position);
                success = true;
            }
            catch (IOException e) {
                cause = e;
            }
        } while (!success && count <= 5);
        if (!success) {
            throw cause;
        }
    }

    public static void truncateFile(File file, long position) throws IOException {
        try (RandomAccessFile access = new RandomAccessFile(file, "rw");){
            FileUtils.truncateFile(access.getChannel(), position);
        }
    }

    private static void waitAndThenTriggerGC() {
        try {
            Thread.sleep(500L);
        }
        catch (InterruptedException ee) {
            Thread.interrupted();
        }
        System.gc();
    }

    public static String fixSeparatorsInPath(String path) {
        String fileSeparator = System.getProperty("file.separator");
        if ("\\".equals(fileSeparator)) {
            path = path.replace('/', '\\');
        } else if ("/".equals(fileSeparator)) {
            path = path.replace('\\', '/');
        }
        return path;
    }

    public static void copyFile(File srcFile, File dstFile) throws IOException {
        dstFile.getParentFile().mkdirs();
        try (FileInputStream input = new FileInputStream(srcFile);
             FileOutputStream output = new FileOutputStream(dstFile);){
            int bytesRead;
            int bufferSize = 1024;
            byte[] buffer = new byte[bufferSize];
            while ((bytesRead = input.read(buffer)) != -1) {
                output.write(buffer, 0, bytesRead);
            }
        }
        catch (IOException e) {
            throw new IOException("Could not copy '" + srcFile + "' to '" + dstFile + "'", e);
        }
    }

    public static void copyRecursively(File fromDirectory, File toDirectory) throws IOException {
        FileUtils.copyRecursively(fromDirectory, toDirectory, null);
    }

    public static void copyRecursively(File fromDirectory, File toDirectory, FileFilter filter) throws IOException {
        for (File fromFile : fromDirectory.listFiles(filter)) {
            File toFile = new File(toDirectory, fromFile.getName());
            if (fromFile.isDirectory()) {
                Files.createDirectories(toFile.toPath(), new FileAttribute[0]);
                FileUtils.copyRecursively(fromFile, toFile, filter);
                continue;
            }
            FileUtils.copyFile(fromFile, toFile);
        }
    }

    public static void writeToFile(File target, String text, boolean append) throws IOException {
        if (!target.exists()) {
            Files.createDirectories(target.getParentFile().toPath(), new FileAttribute[0]);
            target.createNewFile();
        }
        try (OutputStreamWriter out = new OutputStreamWriter((OutputStream)new FileOutputStream(target, append), StandardCharsets.UTF_8);){
            out.write(text);
        }
    }

    public static BufferedReader newBufferedFileReader(File file, Charset charset) throws FileNotFoundException {
        return new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(file), charset));
    }

    public static PrintWriter newFilePrintWriter(File file, Charset charset) throws FileNotFoundException {
        return new PrintWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file, true), charset));
    }

    public static File path(String root, String ... path) {
        return FileUtils.path(new File(root), path);
    }

    public static File path(File root, String ... path) {
        for (String part : path) {
            root = new File(root, part);
        }
        return root;
    }

    public static void windowsSafeIOOperation(FileOperation operation) throws IOException {
        IOException storedIoe = null;
        for (int i = 0; i < 10; ++i) {
            try {
                operation.perform();
                return;
            }
            catch (IOException e) {
                storedIoe = e;
                System.gc();
                continue;
            }
        }
        throw storedIoe;
    }

    public static LineListener echo(final PrintStream target) {
        return new LineListener(){

            @Override
            public void line(String line) {
                target.println(line);
            }
        };
    }

    public static void readTextFile(File file, LineListener listener) throws IOException {
        try (BufferedReader reader = new BufferedReader(new FileReader(file));){
            String line;
            while ((line = reader.readLine()) != null) {
                listener.line(line);
            }
        }
    }

    public static String readTextFile(File file, Charset charset) throws IOException {
        StringBuilder out = new StringBuilder();
        for (String s : Files.readAllLines(file.toPath(), charset)) {
            out.append(s).append("\n");
        }
        return out.toString();
    }

    private static void deleteFileWithRetries(Path file, int tries) throws IOException {
        try {
            Files.delete(file);
        }
        catch (IOException e) {
            if (SystemUtils.IS_OS_WINDOWS && FileUtils.mayBeWindowsMemoryMappedFileReleaseProblem(e)) {
                if (tries >= 5) {
                    throw new MaybeWindowsMemoryMappedFileReleaseProblem(e);
                }
                FileUtils.waitAndThenTriggerGC();
                FileUtils.deleteFileWithRetries(file, tries + 1);
            }
            throw e;
        }
    }

    private static boolean mayBeWindowsMemoryMappedFileReleaseProblem(IOException e) {
        return e.getMessage().contains("The process cannot access the file because it is being used by another process.");
    }

    public static String relativePath(File baseDir, File storeFile) throws IOException {
        String prefix = baseDir.getCanonicalPath();
        String path = storeFile.getCanonicalPath();
        if (!path.startsWith(prefix)) {
            throw new FileNotFoundException();
        }
        if ((path = path.substring(prefix.length())).startsWith(File.separator)) {
            return path.substring(1);
        }
        return path;
    }

    public static File getMostCanonicalFile(File file) {
        try {
            return file.getCanonicalFile().getAbsoluteFile();
        }
        catch (IOException e) {
            return file.getAbsoluteFile();
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src, long position) throws IOException {
        int bytesWritten;
        long filePosition = position;
        long expectedEndPosition = filePosition + (long)src.limit() - (long)src.position();
        while ((filePosition += (long)(bytesWritten = channel.write(src, filePosition))) < expectedEndPosition) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src) throws IOException {
        int bytesWritten;
        long bytesToWrite = src.limit() - src.position();
        while ((bytesToWrite -= (long)(bytesWritten = channel.write(src))) > 0L) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static OpenOption[] convertOpenMode(String mode) {
        OpenOption[] options;
        switch (mode) {
            case "r": {
                options = new OpenOption[]{StandardOpenOption.READ};
                break;
            }
            case "rw": {
                options = new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE};
                break;
            }
            case "rws": {
                options = new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SYNC};
                break;
            }
            case "rwd": {
                options = new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.DSYNC};
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported mode: " + mode);
            }
        }
        return options;
    }

    public static FileChannel open(Path path, String mode) throws IOException {
        return FileChannel.open(path, FileUtils.convertOpenMode(mode));
    }

    public static InputStream openAsInputStream(Path path) throws IOException {
        return Files.newInputStream(path, StandardOpenOption.READ);
    }

    public static boolean isEmptyDirectory(File directory) throws IOException {
        if (directory.exists()) {
            if (!directory.isDirectory()) {
                throw new IllegalArgumentException("Expected directory, but was file: " + directory);
            }
            try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(directory.toPath());){
                boolean bl = !directoryStream.iterator().hasNext();
                return bl;
            }
        }
        return true;
    }

    public static OutputStream openAsOutputStream(Path path, boolean append) throws IOException {
        OpenOption[] options = append ? new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.APPEND} : new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE};
        return Files.newOutputStream(path, options);
    }

    public static class MaybeWindowsMemoryMappedFileReleaseProblem
    extends IOException {
        public MaybeWindowsMemoryMappedFileReleaseProblem(IOException e) {
            super(e);
        }
    }

    public static interface LineListener {
        public void line(String var1);
    }

    public static interface FileOperation {
        public void perform() throws IOException;
    }
}

