/**
 * This file is part of Graylog2.
 *
 * Graylog2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Graylog2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Graylog2.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.graylog2.radio.rest.resources.system;

import com.codahale.metrics.*;
import com.codahale.metrics.annotation.Timed;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.graylog2.radio.rest.resources.RestResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import java.util.List;
import java.util.Map;

@Path("/system/metrics")
public class MetricsResource extends RestResource {

    private static final Logger LOG = LoggerFactory.getLogger(MetricsResource.class);
    
    @Inject
    private MetricRegistry metricRegistry;

    @GET
    @Timed
    @Produces(MediaType.APPLICATION_JSON)
    public MetricRegistry metrics() {
        return metricRegistry;
    }

    @GET @Timed
    @Path("/names")
    @Produces(MediaType.APPLICATION_JSON)
    public String metricNames() {
        Map<String, Object> result = Maps.newHashMap();
        result.put("names", metricRegistry.getNames());

        return json(result);
    }

    @GET @Timed
    @Path("/{metricName}")
    @Produces(MediaType.APPLICATION_JSON)
    public Metric singleMetric(@PathParam("metricName") String metricName) {
        Metric metric = metricRegistry.getMetrics().get(metricName);

        if (metric == null) {
            LOG.debug("I do not have a metric called [{}], returning 404.", metricName);
            throw new WebApplicationException(404);
        }

        return metric;
    }

    @POST @Timed
    @Path("/multiple")
    public String multipleMetrics(MetricsReadRequest request) {
        final Map<String, Metric> metrics = metricRegistry.getMetrics();

        List<Map<String, Object>> metricsList = Lists.newArrayList();
        if (request.metrics == null) {
            throw new BadRequestException("Metrics cannot be empty");
        }

        for (String name : request.metrics) {
            final Metric metric = metrics.get(name);
            if (metric != null) {
                metricsList.add(getMetricMap(name, metric));
            }
        }

        Map<String, Object> result = Maps.newHashMap();
        result.put("metrics", metricsList);
        result.put("total", metricsList.size());
        return json(result);
    }

    private Map<String, Object> getMetricMap(String metricName, Metric metric) {
        String type = metric.getClass().getSimpleName().toLowerCase();

        if (type.isEmpty()) {
            type = "gauge";
        }

        Map<String, Object> metricMap = Maps.newHashMap();
        metricMap.put("full_name", metricName);
        metricMap.put("name", metricName.substring(metricName.lastIndexOf(".") + 1));
        metricMap.put("type", type);

        if (metric instanceof Timer) {
            metricMap.put("metric", buildTimerMap((Timer) metric));
        } else if(metric instanceof Meter) {
            metricMap.put("metric", buildMeterMap((Meter) metric));
        } else if(metric instanceof Histogram) {
            metricMap.put("metric", buildHistogramMap((Histogram) metric));
        } else {
            metricMap.put("metric", metric);
        }
        return metricMap;
    }

    @GET @Timed
    @Path("/namespace/{namespace}")
    @Produces(MediaType.APPLICATION_JSON)
    public String byNamespace(@PathParam("namespace") String namespace) {
        List<Map<String, Object>> metrics = Lists.newArrayList();

        for(Map.Entry<String, Metric> e : metricRegistry.getMetrics().entrySet()) {
            if (e.getKey().startsWith(namespace)) {
                try {
                    String type = e.getValue().getClass().getSimpleName().toLowerCase();
                    String metricName = e.getKey();

                    if (type.isEmpty()) {
                        type = "gauge";
                    }

                    Map<String, Object> metric = Maps.newHashMap();
                    metric.put("full_name", metricName);
                    metric.put("name", metricName.substring(metricName.lastIndexOf(".") + 1));
                    metric.put("type", type);

                    if (e.getValue() instanceof Timer) {
                        metric.put("metric", buildTimerMap((Timer) e.getValue()));
                    } else if(e.getValue() instanceof Meter) {
                        metric.put("metric", buildMeterMap((Meter) e.getValue()));
                    } else if(e.getValue() instanceof Histogram) {
                        metric.put("metric", buildHistogramMap((Histogram) e.getValue()));
                    } else {
                        metric.put("metric", e.getValue());
                    }

                    metrics.add(metric);
                } catch(Exception ex) {
                    LOG.warn("Could not read metric in namespace list.", ex);
                    continue;
                }
            }
        }

        if (metrics.isEmpty()) {
            LOG.debug("No metrics with namespace [{}] found, returning 404.", namespace);
            throw new WebApplicationException(404);
        }

        Map<String, Object> result = Maps.newHashMap();
        result.put("metrics", metrics);
        result.put("total", metrics.size());

        return json(result);
    }


    @JsonAutoDetect
    public static class MetricsReadRequest {

        public String[] metrics;
    }
}