/**
 * This file is part of Graylog2.
 *
 * Graylog2 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Graylog2 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Graylog2.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.graylog2.radio.transports.amqp;

import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import org.graylog2.plugin.Message;
import org.graylog2.radio.Configuration;
import org.graylog2.radio.transports.RadioTransport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import java.io.IOException;

import static com.codahale.metrics.MetricRegistry.name;

/**
 * @author Lennart Koopmann <lennart@torch.sh>
 */
public class AMQPProducer implements RadioTransport {

    private static final Logger LOG = LoggerFactory.getLogger(AMQPProducer.class);

    public final static String EXCHANGE = "graylog2";
    public final static String QUEUE = "graylog2-radio-messages";
    public final static String ROUTING_KEY = "graylog2-radio-message";

    private final AMQPSender sender;
    private final MetricRegistry metricRegistry;
    private final Meter incomingMessages;
    private final Meter rejectedMessages;
    private final Timer processTime;

    @Inject
    public AMQPProducer(Configuration configuration, MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
        sender = new AMQPSender(
                configuration.getAmqpHostname(),
                configuration.getAmqpPort(),
                configuration.getAmqpVirtualHost(),
                configuration.getAmqpUsername(),
                configuration.getAmqpPassword()
        );
        incomingMessages = metricRegistry.meter(name(AMQPProducer.class, "incomingMessages"));
        rejectedMessages = metricRegistry.meter(name(AMQPProducer.class, "rejectedMessages"));
        processTime = metricRegistry.timer(name(AMQPProducer.class, "processTime"));
    }

    @Override
    public void send(Message msg) {
        try (Timer.Context context = processTime.time()) {
            incomingMessages.mark();
            sender.send(msg);
        } catch (IOException e) {
            LOG.error("Could not write to AMQP.", e);
            rejectedMessages.mark();
        }
    }

}
