/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.inputs.syslog;

import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.google.common.collect.Maps;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Map;
import java.util.regex.Pattern;
import org.graylog2.inputs.syslog.StructuredSyslog;
import org.graylog2.plugin.Message;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.buffers.Buffer;
import org.graylog2.plugin.buffers.BufferOutOfCapacityException;
import org.graylog2.plugin.configuration.Configuration;
import org.graylog2.plugin.inputs.MessageInput;
import org.graylog2.syslog4j.server.SyslogServerEventIF;
import org.graylog2.syslog4j.server.impl.event.SyslogServerEvent;
import org.graylog2.syslog4j.server.impl.event.structured.StructuredSyslogServerEvent;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SyslogProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(SyslogProcessor.class);
    private final Buffer processBuffer;
    private final Configuration config;
    private final MessageInput sourceInput;
    private static final Pattern STRUCTURED_SYSLOG_PATTERN = Pattern.compile("<\\d+>\\d.*", 32);
    private final Meter incomingMessages;
    private final Meter parsingFailures;
    private final Meter incompleteMessages;
    private final Meter processedMessages;
    private final Timer syslogParsedTime;

    public SyslogProcessor(MetricRegistry metricRegistry, Buffer processBuffer, Configuration config, MessageInput sourceInput) {
        this.processBuffer = processBuffer;
        this.config = config;
        this.sourceInput = sourceInput;
        String metricsId = sourceInput.getUniqueReadableId();
        this.incomingMessages = metricRegistry.meter(MetricRegistry.name(metricsId, "incomingMessages"));
        this.parsingFailures = metricRegistry.meter(MetricRegistry.name(metricsId, "parsingFailures"));
        this.processedMessages = metricRegistry.meter(MetricRegistry.name(metricsId, "processedMessages"));
        this.incompleteMessages = metricRegistry.meter(MetricRegistry.name(metricsId, "incompleteMessages"));
        this.syslogParsedTime = metricRegistry.timer(MetricRegistry.name(metricsId, "syslogParsedTime"));
    }

    public void messageReceived(String msg, InetAddress remoteAddress) throws BufferOutOfCapacityException {
        Message lm;
        this.incomingMessages.mark();
        try {
            lm = this.parse(msg, remoteAddress);
        }
        catch (Exception e) {
            this.parsingFailures.mark();
            LOG.warn("Could not parse syslog message. Not further handling.", e);
            return;
        }
        if (!lm.isComplete()) {
            this.incompleteMessages.mark();
            LOG.debug("Skipping incomplete message. Parsed fields: [{}]", (Object)lm.getFields());
            return;
        }
        LOG.debug("Adding received syslog message <{}> to process buffer: {}", (Object)lm.getId(), (Object)lm);
        this.processedMessages.mark();
        this.processBuffer.insertCached(lm, this.sourceInput);
    }

    private Message parse(String msg, InetAddress remoteAddress) throws UnknownHostException {
        Timer.Context tcx = this.syslogParsedTime.time();
        if (remoteAddress == null) {
            remoteAddress = InetAddress.getLocalHost();
        }
        SyslogServerEvent e = SyslogProcessor.isStructuredSyslog(msg) ? new StructuredSyslogServerEvent(msg, remoteAddress) : new SyslogServerEvent(msg, remoteAddress);
        Message m = new Message(e.getMessage(), this.parseHost(e, remoteAddress), this.parseDate(e));
        m.addField("facility", Tools.syslogFacilityToReadable(e.getFacility()));
        m.addField("level", e.getLevel());
        if (this.config.getBoolean("store_full_message")) {
            m.addField("full_message", new String(e.getRaw()));
        }
        m.addFields(this.parseAdditionalData(e));
        tcx.stop();
        return m;
    }

    private Map<String, Object> parseAdditionalData(SyslogServerEventIF msg) {
        Map<String, Object> structuredData = Maps.newHashMap();
        if (msg instanceof StructuredSyslogServerEvent) {
            StructuredSyslogServerEvent sMsg = (StructuredSyslogServerEvent)msg;
            structuredData = StructuredSyslog.extractFields(sMsg);
            if (sMsg.getApplicationName() != null && !sMsg.getApplicationName().isEmpty()) {
                structuredData.put("application_name", sMsg.getApplicationName());
            }
            if (sMsg.getProcessId() != null && !sMsg.getProcessId().isEmpty()) {
                structuredData.put("process_id", sMsg.getProcessId());
            }
        }
        return structuredData;
    }

    private String parseHost(SyslogServerEventIF msg, InetAddress remoteAddress) {
        if (remoteAddress != null && this.config.getBoolean("force_rdns")) {
            try {
                return Tools.rdnsLookup(remoteAddress);
            }
            catch (UnknownHostException e) {
                LOG.warn("Reverse DNS lookup failed. Falling back to parsed hostname.", e);
            }
        }
        return msg.getHost();
    }

    private DateTime parseDate(SyslogServerEventIF msg) throws IllegalStateException {
        if (msg.getDate() == null) {
            if (this.config.getBoolean("allow_override_date")) {
                LOG.debug("Date could not be parsed. Was set to NOW because {} is true.", (Object)"allow_override_date");
                return Tools.iso8601();
            }
            LOG.warn("Syslog message is missing date or date could not be parsed. (Possibly set {} to true) Not further handling. Message was: {}", (Object)"allow_override_date", (Object)new String(msg.getRaw()));
            throw new IllegalStateException();
        }
        return new DateTime(msg.getDate());
    }

    public static boolean isStructuredSyslog(String message) {
        return STRUCTURED_SYSLOG_PATTERN.matcher(message).matches();
    }
}

