/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.graylog.shaded.opensearch22.org.opensearch.index.engine;

import org.graylog.shaded.opensearch22.org.apache.lucene.index.LeafReader;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.NumericDocValues;
import org.graylog.shaded.opensearch22.org.opensearch.index.mapper.SeqNoFieldMapper;
import org.graylog.shaded.opensearch22.org.opensearch.index.mapper.SourceFieldMapper;
import org.graylog.shaded.opensearch22.org.opensearch.index.mapper.VersionFieldMapper;

import java.io.IOException;
import java.util.Objects;

/**
 * Doc Values used in LuceneChangesSnapshot for recovery
 *
 * @opensearch.internal
 */
final class CombinedDocValues {
    private final NumericDocValues versionDV;
    private final NumericDocValues seqNoDV;
    private final NumericDocValues primaryTermDV;
    private final NumericDocValues tombstoneDV;
    private final NumericDocValues recoverySource;

    CombinedDocValues(LeafReader leafReader) throws IOException {
        this.versionDV = Objects.requireNonNull(leafReader.getNumericDocValues(VersionFieldMapper.NAME), "VersionDV is missing");
        this.seqNoDV = Objects.requireNonNull(leafReader.getNumericDocValues(SeqNoFieldMapper.NAME), "SeqNoDV is missing");
        this.primaryTermDV = Objects.requireNonNull(
            leafReader.getNumericDocValues(SeqNoFieldMapper.PRIMARY_TERM_NAME),
            "PrimaryTermDV is missing"
        );
        this.tombstoneDV = leafReader.getNumericDocValues(SeqNoFieldMapper.TOMBSTONE_NAME);
        this.recoverySource = leafReader.getNumericDocValues(SourceFieldMapper.RECOVERY_SOURCE_NAME);
    }

    long docVersion(int segmentDocId) throws IOException {
        assert versionDV.docID() < segmentDocId;
        if (versionDV.advanceExact(segmentDocId) == false) {
            assert false : "DocValues for field [" + VersionFieldMapper.NAME + "] is not found";
            throw new IllegalStateException("DocValues for field [" + VersionFieldMapper.NAME + "] is not found");
        }
        return versionDV.longValue();
    }

    long docSeqNo(int segmentDocId) throws IOException {
        assert seqNoDV.docID() < segmentDocId;
        if (seqNoDV.advanceExact(segmentDocId) == false) {
            assert false : "DocValues for field [" + SeqNoFieldMapper.NAME + "] is not found";
            throw new IllegalStateException("DocValues for field [" + SeqNoFieldMapper.NAME + "] is not found");
        }
        return seqNoDV.longValue();
    }

    long docPrimaryTerm(int segmentDocId) throws IOException {
        // We exclude non-root nested documents when querying changes, every returned document must have primary term.
        assert primaryTermDV.docID() < segmentDocId;
        if (primaryTermDV.advanceExact(segmentDocId) == false) {
            assert false : "DocValues for field [" + SeqNoFieldMapper.PRIMARY_TERM_NAME + "] is not found";
            throw new IllegalStateException("DocValues for field [" + SeqNoFieldMapper.PRIMARY_TERM_NAME + "] is not found");
        }
        return primaryTermDV.longValue();
    }

    boolean isTombstone(int segmentDocId) throws IOException {
        if (tombstoneDV == null) {
            return false;
        }
        assert tombstoneDV.docID() < segmentDocId;
        return tombstoneDV.advanceExact(segmentDocId) && tombstoneDV.longValue() > 0;
    }

    boolean hasRecoverySource(int segmentDocId) throws IOException {
        if (recoverySource == null) {
            return false;
        }
        assert recoverySource.docID() < segmentDocId;
        return recoverySource.advanceExact(segmentDocId);
    }
}
