/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
Copyright (C) 2007 Google Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.graylog.shaded.opensearch22.org.opensearch.common.inject.internal;

import org.graylog.shaded.opensearch22.org.opensearch.common.inject.Binder;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.Injector;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.Key;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.Provider;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.spi.BindingTargetVisitor;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.spi.Dependency;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.spi.HasDependencies;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.spi.InjectionPoint;
import org.graylog.shaded.opensearch22.org.opensearch.common.inject.spi.ProviderInstanceBinding;

import java.util.HashSet;
import java.util.Set;

import static java.util.Collections.unmodifiableSet;

/**
 * Provider instance binding
 *
 * @opensearch.internal
 */
public final class ProviderInstanceBindingImpl<T> extends BindingImpl<T> implements ProviderInstanceBinding<T> {

    final Provider<? extends T> providerInstance;
    final Set<InjectionPoint> injectionPoints;

    public ProviderInstanceBindingImpl(
        Injector injector,
        Key<T> key,
        Object source,
        InternalFactory<? extends T> internalFactory,
        Scoping scoping,
        Provider<? extends T> providerInstance,
        Set<InjectionPoint> injectionPoints
    ) {
        super(injector, key, source, internalFactory, scoping);
        this.providerInstance = providerInstance;
        this.injectionPoints = injectionPoints;
    }

    public ProviderInstanceBindingImpl(
        Object source,
        Key<T> key,
        Scoping scoping,
        Set<InjectionPoint> injectionPoints,
        Provider<? extends T> providerInstance
    ) {
        super(source, key, scoping);
        this.injectionPoints = injectionPoints;
        this.providerInstance = providerInstance;
    }

    @Override
    public <V> V acceptTargetVisitor(BindingTargetVisitor<? super T, V> visitor) {
        return visitor.visit(this);
    }

    @Override
    public Provider<? extends T> getProviderInstance() {
        return providerInstance;
    }

    @Override
    public Set<InjectionPoint> getInjectionPoints() {
        return injectionPoints;
    }

    @Override
    public Set<Dependency<?>> getDependencies() {
        return providerInstance instanceof HasDependencies
            ? unmodifiableSet(new HashSet<>((((HasDependencies) providerInstance).getDependencies())))
            : Dependency.forInjectionPoints(injectionPoints);
    }

    @Override
    public BindingImpl<T> withScoping(Scoping scoping) {
        return new ProviderInstanceBindingImpl<>(getSource(), getKey(), scoping, injectionPoints, providerInstance);
    }

    @Override
    public BindingImpl<T> withKey(Key<T> key) {
        return new ProviderInstanceBindingImpl<>(getSource(), key, getScoping(), injectionPoints, providerInstance);
    }

    @Override
    public void applyTo(Binder binder) {
        getScoping().applyTo(binder.withSource(getSource()).bind(getKey()).toProvider(getProviderInstance()));
    }

    @Override
    public String toString() {
        return new ToStringBuilder(ProviderInstanceBinding.class).add("key", getKey())
            .add("source", getSource())
            .add("scope", getScoping())
            .add("provider", providerInstance)
            .toString();
    }
}
