/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.graylog.shaded.opensearch22.org.opensearch.action;

import org.graylog.shaded.opensearch22.org.opensearch.common.CheckedConsumer;
import org.graylog.shaded.opensearch22.org.opensearch.common.CheckedRunnable;
import org.graylog.shaded.opensearch22.org.opensearch.common.CheckedSupplier;
import org.graylog.shaded.opensearch22.org.opensearch.common.util.concurrent.AbstractRunnable;

/**
 * Base class for {@link Runnable}s that need to call {@link ActionListener#onFailure(Exception)} in case an uncaught
 * exception or error is thrown while the actual action is run.
 *
 * @opensearch.internal
 */
public abstract class ActionRunnable<Response> extends AbstractRunnable {

    protected final ActionListener<Response> listener;

    /**
     * Creates a {@link Runnable} that invokes the given listener with {@code null} after the given runnable has executed.
     * @param listener Listener to invoke
     * @param runnable Runnable to execute
     * @return Wrapped {@code Runnable}
     */
    public static <T> ActionRunnable<T> run(ActionListener<T> listener, CheckedRunnable<Exception> runnable) {
        return new ActionRunnable<T>(listener) {
            @Override
            protected void doRun() throws Exception {
                runnable.run();
                listener.onResponse(null);
            }
        };
    }

    /**
     * Creates a {@link Runnable} that invokes the given listener with the return of the given supplier.
     * @param listener Listener to invoke
     * @param supplier Supplier that provides value to pass to listener
     * @return Wrapped {@code Runnable}
     */
    public static <T> ActionRunnable<T> supply(ActionListener<T> listener, CheckedSupplier<T, Exception> supplier) {
        return ActionRunnable.wrap(listener, l -> l.onResponse(supplier.get()));
    }

    /**
     * Creates a {@link Runnable} that wraps the given listener and a consumer of it that is executed when the {@link Runnable} is run.
     * Invokes {@link ActionListener#onFailure(Exception)} on it if an exception is thrown on executing the consumer.
     * @param listener ActionListener to wrap
     * @param consumer Consumer of wrapped {@code ActionListener}
     * @param <T> Type of the given {@code ActionListener}
     * @return Wrapped {@code Runnable}
     */
    public static <T> ActionRunnable<T> wrap(ActionListener<T> listener, CheckedConsumer<ActionListener<T>, Exception> consumer) {
        return new ActionRunnable<T>(listener) {
            @Override
            protected void doRun() throws Exception {
                consumer.accept(listener);
            }
        };
    }

    public ActionRunnable(ActionListener<Response> listener) {
        this.listener = listener;
    }

    /**
     * Calls the action listeners {@link ActionListener#onFailure(Exception)} method with the given exception.
     * This method is invoked for all exception thrown by {@link #doRun()}
     */
    @Override
    public void onFailure(Exception e) {
        listener.onFailure(e);
    }
}
