/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.graylog.shaded.opensearch22.org.apache.lucene.search.grouping;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.LeafReaderContext;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.DoubleValuesSource;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.LongValues;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.LongValuesSource;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.Scorable;

/** A GroupSelector implementation that groups documents by long values */
public class LongRangeGroupSelector extends GroupSelector<LongRange> {

  private final LongValuesSource source;
  private final LongRangeFactory rangeFactory;

  private Set<LongRange> inSecondPass;
  private boolean includeEmpty = true;
  private boolean positioned;
  private LongRange current;

  private LeafReaderContext context;
  private LongValues values;

  /**
   * Creates a new LongRangeGroupSelector
   *
   * @param source a LongValuesSource to retrieve long values per document
   * @param rangeFactory a LongRangeFactory that defines how to group the long values into range
   *     buckets
   */
  public LongRangeGroupSelector(LongValuesSource source, LongRangeFactory rangeFactory) {
    this.source = source;
    this.rangeFactory = rangeFactory;
  }

  @Override
  public void setNextReader(LeafReaderContext readerContext) throws IOException {
    this.context = readerContext;
  }

  @Override
  public void setScorer(Scorable scorer) throws IOException {
    this.values = source.getValues(context, DoubleValuesSource.fromScorer(scorer));
  }

  @Override
  public State advanceTo(int doc) throws IOException {
    positioned = values.advanceExact(doc);
    if (positioned == false) {
      return includeEmpty ? State.ACCEPT : State.SKIP;
    }
    this.current = rangeFactory.getRange(values.longValue(), this.current);
    if (inSecondPass == null) {
      return State.ACCEPT;
    }
    return inSecondPass.contains(this.current) ? State.ACCEPT : State.SKIP;
  }

  @Override
  public LongRange currentValue() throws IOException {
    return positioned ? this.current : null;
  }

  @Override
  public LongRange copyValue() throws IOException {
    return positioned ? new LongRange(this.current.min, this.current.max) : null;
  }

  @Override
  public void setGroups(Collection<SearchGroup<LongRange>> searchGroups) {
    inSecondPass = new HashSet<>();
    includeEmpty = false;
    for (SearchGroup<LongRange> group : searchGroups) {
      if (group.groupValue == null) {
        includeEmpty = true;
      } else {
        inSecondPass.add(group.groupValue);
      }
    }
  }
}
