/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.graylog.shaded.opensearch22.org.apache.lucene.queries.spans;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.TreeMap;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.Term;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.TermStates;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.IndexSearcher;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.Query;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.ScoreMode;

/** Base class for span-based queries. */
public abstract class SpanQuery extends Query {

  /** Returns the name of the field matched by this query. */
  public abstract String getField();

  @Override
  public abstract SpanWeight createWeight(IndexSearcher searcher, ScoreMode scoreMode, float boost)
      throws IOException;

  /**
   * Build a map of terms to {@link TermStates}, for use in constructing SpanWeights
   *
   * @lucene.internal
   */
  public static Map<Term, TermStates> getTermStates(SpanWeight... weights) {
    Map<Term, TermStates> terms = new TreeMap<>();
    for (SpanWeight w : weights) {
      w.extractTermStates(terms);
    }
    return terms;
  }

  /**
   * Build a map of terms to {@link TermStates}, for use in constructing SpanWeights
   *
   * @lucene.internal
   */
  public static Map<Term, TermStates> getTermStates(Collection<SpanWeight> weights) {
    Map<Term, TermStates> terms = new TreeMap<>();
    for (SpanWeight w : weights) {
      w.extractTermStates(terms);
    }
    return terms;
  }
}
