/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.graylog.shaded.opensearch22.org.apache.lucene.queries.function.valuesource;

import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.DocValues;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.LeafReaderContext;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.NumericDocValues;
import org.graylog.shaded.opensearch22.org.apache.lucene.index.SortedNumericDocValues;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.SortField;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.SortedNumericSelector;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.SortedNumericSelector.Type;
import org.graylog.shaded.opensearch22.org.apache.lucene.search.SortedNumericSortField;

/**
 * Obtains long field values from {@link
 * org.graylog.shaded.opensearch22.org.apache.lucene.index.LeafReader#getSortedNumericDocValues} and using a {@link
 * org.graylog.shaded.opensearch22.org.apache.lucene.search.SortedNumericSelector} it gives a single-valued ValueSource view of a
 * field.
 */
public class MultiValuedLongFieldSource extends LongFieldSource {

  protected final SortedNumericSelector.Type selector;

  public MultiValuedLongFieldSource(String field, Type selector) {
    super(field);
    this.selector = selector;
    Objects.requireNonNull(field, "Field is required to create a MultiValuedLongFieldSource");
    Objects.requireNonNull(
        selector, "SortedNumericSelector is required to create a MultiValuedLongFieldSource");
  }

  @Override
  public SortField getSortField(boolean reverse) {
    return new SortedNumericSortField(field, SortField.Type.LONG, reverse, selector);
  }

  @Override
  public String description() {
    return "long(" + field + ',' + selector.name() + ')';
  }

  @Override
  protected NumericDocValues getNumericDocValues(
      Map<Object, Object> context, LeafReaderContext readerContext) throws IOException {
    SortedNumericDocValues sortedDv = DocValues.getSortedNumeric(readerContext.reader(), field);
    return SortedNumericSelector.wrap(sortedDv, selector, SortField.Type.LONG);
  }

  @Override
  public boolean equals(Object o) {
    if (o.getClass() != MultiValuedLongFieldSource.class) return false;
    MultiValuedLongFieldSource other = (MultiValuedLongFieldSource) o;
    if (this.selector != other.selector) return false;
    return this.field.equals(other.field);
  }

  @Override
  public int hashCode() {
    int h = super.hashCode();
    h += selector.hashCode();
    return h;
  }
}
