/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.graylog.shaded.opensearch22.org.apache.lucene.analysis.el;

import java.io.IOException;
import java.io.Reader;
import java.io.UncheckedIOException;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.Analyzer;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.CharArraySet;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.StopFilter;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.StopwordAnalyzerBase;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.TokenStream;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.Tokenizer;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.WordlistLoader;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.standard.StandardTokenizer;
import org.graylog.shaded.opensearch22.org.apache.lucene.util.IOUtils;

/**
 * {@link Analyzer} for the Greek language.
 *
 * <p>Supports an external list of stopwords (words that will not be indexed at all). A default set
 * of stopwords is used unless an alternative list is specified.
 *
 * <p><b>NOTE</b>: This class uses the same {@link org.graylog.shaded.opensearch22.org.apache.lucene.util.Version} dependent
 * settings as {@link StandardAnalyzer}.
 *
 * @since 3.1
 */
public final class GreekAnalyzer extends StopwordAnalyzerBase {
  /** File containing default Greek stopwords. */
  public static final String DEFAULT_STOPWORD_FILE = "stopwords.txt";

  /**
   * Returns a set of default Greek-stopwords
   *
   * @return a set of default Greek-stopwords
   */
  public static final CharArraySet getDefaultStopSet() {
    return DefaultSetHolder.DEFAULT_SET;
  }

  private static class DefaultSetHolder {
    private static final CharArraySet DEFAULT_SET;

    static {
      try {
        DEFAULT_SET =
            WordlistLoader.getWordSet(
                IOUtils.requireResourceNonNull(
                    GreekAnalyzer.class.getResourceAsStream(DEFAULT_STOPWORD_FILE),
                    DEFAULT_STOPWORD_FILE),
                "#");
      } catch (IOException ex) {
        // default set should always be present as it is part of the
        // distribution (JAR)
        throw new UncheckedIOException("Unable to load default stopword set", ex);
      }
    }
  }

  /** Builds an analyzer with the default stop words. */
  public GreekAnalyzer() {
    this(DefaultSetHolder.DEFAULT_SET);
  }

  /**
   * Builds an analyzer with the given stop words.
   *
   * <p><b>NOTE:</b> The stopwords set should be pre-processed with the logic of {@link
   * GreekLowerCaseFilter} for best results.
   *
   * @param stopwords a stopword set
   */
  public GreekAnalyzer(CharArraySet stopwords) {
    super(stopwords);
  }

  /**
   * Creates {@link org.graylog.shaded.opensearch22.org.apache.lucene.analysis.Analyzer.TokenStreamComponents} used to tokenize all
   * the text in the provided {@link Reader}.
   *
   * @return {@link org.graylog.shaded.opensearch22.org.apache.lucene.analysis.Analyzer.TokenStreamComponents} built from a {@link
   *     StandardTokenizer} filtered with {@link GreekLowerCaseFilter}, {@link StopFilter}, and
   *     {@link GreekStemFilter}
   */
  @Override
  protected TokenStreamComponents createComponents(String fieldName) {
    final Tokenizer source = new StandardTokenizer();
    TokenStream result = new GreekLowerCaseFilter(source);
    result = new StopFilter(result, stopwords);
    result = new GreekStemFilter(result);
    return new TokenStreamComponents(source, result);
  }

  @Override
  protected TokenStream normalize(String fieldName, TokenStream in) {
    return new GreekLowerCaseFilter(in);
  }
}
