/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.graylog.shaded.opensearch22.org.apache.lucene.analysis.core;

import java.util.Map;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.TokenFilterFactory;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.TokenStream;

/**
 * Factory for {@link UpperCaseFilter}.
 *
 * <pre class="prettyprint">
 * &lt;fieldType name="text_uppercase" class="solr.TextField" positionIncrementGap="100"&gt;
 *   &lt;analyzer&gt;
 *     &lt;tokenizer class="solr.WhitespaceTokenizerFactory"/&gt;
 *     &lt;filter class="solr.UpperCaseFilterFactory"/&gt;
 *   &lt;/analyzer&gt;
 * &lt;/fieldType&gt;</pre>
 *
 * <p><b>NOTE:</b> In Unicode, this transformation may lose information when the upper case
 * character represents more than one lower case character. Use this filter when you require
 * uppercase tokens. Use the {@link LowerCaseFilterFactory} for general search matching
 *
 * @since 4.7.0
 * @lucene.spi {@value #NAME}
 */
public class UpperCaseFilterFactory extends TokenFilterFactory {

  /** SPI name */
  public static final String NAME = "uppercase";

  /** Creates a new UpperCaseFilterFactory */
  public UpperCaseFilterFactory(Map<String, String> args) {
    super(args);
    if (!args.isEmpty()) {
      throw new IllegalArgumentException("Unknown parameters: " + args);
    }
  }

  /** Default ctor for compatibility with SPI */
  public UpperCaseFilterFactory() {
    throw defaultCtorException();
  }

  @Override
  public TokenStream create(TokenStream input) {
    return new UpperCaseFilter(input);
  }

  @Override
  public TokenStream normalize(TokenStream input) {
    return create(input);
  }
}
