/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.graylog.shaded.opensearch22.org.apache.lucene.analysis.compound;

import java.io.IOException;
import java.util.Map;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.CharArraySet;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.TokenFilterFactory;
import org.graylog.shaded.opensearch22.org.apache.lucene.analysis.TokenStream;
import org.graylog.shaded.opensearch22.org.apache.lucene.util.ResourceLoader;
import org.graylog.shaded.opensearch22.org.apache.lucene.util.ResourceLoaderAware;

/**
 * Factory for {@link DictionaryCompoundWordTokenFilter}.
 *
 * <pre class="prettyprint">
 * &lt;fieldType name="text_dictcomp" class="solr.TextField" positionIncrementGap="100"&gt;
 *   &lt;analyzer&gt;
 *     &lt;tokenizer class="solr.WhitespaceTokenizerFactory"/&gt;
 *     &lt;filter class="solr.DictionaryCompoundWordTokenFilterFactory" dictionary="dictionary.txt"
 *         minWordSize="5" minSubwordSize="2" maxSubwordSize="15" onlyLongestMatch="true"/&gt;
 *   &lt;/analyzer&gt;
 * &lt;/fieldType&gt;</pre>
 *
 * @since 3.1
 * @lucene.spi {@value #NAME}
 */
public class DictionaryCompoundWordTokenFilterFactory extends TokenFilterFactory
    implements ResourceLoaderAware {

  /** SPI name */
  public static final String NAME = "dictionaryCompoundWord";

  private CharArraySet dictionary;
  private final String dictFile;
  private final int minWordSize;
  private final int minSubwordSize;
  private final int maxSubwordSize;
  private final boolean onlyLongestMatch;

  /** Creates a new DictionaryCompoundWordTokenFilterFactory */
  public DictionaryCompoundWordTokenFilterFactory(Map<String, String> args) {
    super(args);
    dictFile = require(args, "dictionary");
    minWordSize = getInt(args, "minWordSize", CompoundWordTokenFilterBase.DEFAULT_MIN_WORD_SIZE);
    minSubwordSize =
        getInt(args, "minSubwordSize", CompoundWordTokenFilterBase.DEFAULT_MIN_SUBWORD_SIZE);
    maxSubwordSize =
        getInt(args, "maxSubwordSize", CompoundWordTokenFilterBase.DEFAULT_MAX_SUBWORD_SIZE);
    onlyLongestMatch = getBoolean(args, "onlyLongestMatch", true);
    if (!args.isEmpty()) {
      throw new IllegalArgumentException("Unknown parameters: " + args);
    }
  }

  /** Default ctor for compatibility with SPI */
  public DictionaryCompoundWordTokenFilterFactory() {
    throw defaultCtorException();
  }

  @Override
  public void inform(ResourceLoader loader) throws IOException {
    dictionary = super.getWordSet(loader, dictFile, false);
  }

  @Override
  public TokenStream create(TokenStream input) {
    // if the dictionary is null, it means it was empty
    if (dictionary == null) {
      return input;
    }
    return new DictionaryCompoundWordTokenFilter(
        input, dictionary, minWordSize, minSubwordSize, maxSubwordSize, onlyLongestMatch);
  }
}
