/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiTimerInfo extends StructStorage<JvmtiTimerInfo.JvmtiTimerInfoWrapper> {

    final int max_value;

    final int may_skip_forward;

    final int may_skip_backward;

    final int kind;

    final int reserved1;

    final int reserved2;

    JvmtiTimerInfo(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiTimerInfo"));
        max_value = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "max_value");
        may_skip_forward = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "may_skip_forward");
        may_skip_backward = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "may_skip_backward");
        kind = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "kind");
        reserved1 = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "reserved1");
        reserved2 = (int) offsetGetter.getOffset("_jvmtiTimerInfo", "reserved2");
    }

    public final class JvmtiTimerInfoWrapper extends StructWrapper {

        JvmtiTimerInfoWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public long maxValue() {
            return getLong(max_value);
        }

        public void maxValue(long valueToPut) {
            putLong(max_value, valueToPut);
        }

        public boolean maySkipForward() {
            return getBoolean(may_skip_forward);
        }

        public void maySkipForward(boolean valueToPut) {
            putBoolean(may_skip_forward, valueToPut);
        }

        public boolean maySkipBackward() {
            return getBoolean(may_skip_backward);
        }

        public void maySkipBackward(boolean valueToPut) {
            putBoolean(may_skip_backward, valueToPut);
        }

        public int kind() {
            return getInt(kind);
        }

        public void kind(int valueToPut) {
            putInt(kind, valueToPut);
        }

        public long reserved1() {
            return getLong(reserved1);
        }

        public void reserved1(long valueToPut) {
            putLong(reserved1, valueToPut);
        }

        public long reserved2() {
            return getLong(reserved2);
        }

        public void reserved2(long valueToPut) {
            putLong(reserved2, valueToPut);
        }

    }

    @Override
    public JvmtiTimerInfoWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiTimerInfoWrapper(jni, pointer);
    }

}

