/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiThreadInfo extends StructStorage<JvmtiThreadInfo.JvmtiThreadInfoWrapper> {

    final int name;

    final int priority;

    final int is_daemon;

    final int thread_group;

    final int context_class_loader;

    JvmtiThreadInfo(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiThreadInfo"));
        name = (int) offsetGetter.getOffset("_jvmtiThreadInfo", "name");
        priority = (int) offsetGetter.getOffset("_jvmtiThreadInfo", "priority");
        is_daemon = (int) offsetGetter.getOffset("_jvmtiThreadInfo", "is_daemon");
        thread_group = (int) offsetGetter.getOffset("_jvmtiThreadInfo", "thread_group");
        context_class_loader = (int) offsetGetter.getOffset("_jvmtiThreadInfo", "context_class_loader");
    }

    public final class JvmtiThreadInfoWrapper extends StructWrapper {

        JvmtiThreadInfoWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public TruffleObject name() {
            return getPointer(name);
        }

        public void name(TruffleObject valueToPut) {
            putPointer(name, valueToPut);
        }

        public int priority() {
            return getInt(priority);
        }

        public void priority(int valueToPut) {
            putInt(priority, valueToPut);
        }

        public boolean isDaemon() {
            return getBoolean(is_daemon);
        }

        public void isDaemon(boolean valueToPut) {
            putBoolean(is_daemon, valueToPut);
        }

        public StaticObject threadGroup() {
            return getObject(thread_group);
        }

        public void threadGroup(StaticObject valueToPut) {
            putObject(thread_group, valueToPut);
        }

        public StaticObject contextClassLoader() {
            return getObject(context_class_loader);
        }

        public void contextClassLoader(StaticObject valueToPut) {
            putObject(context_class_loader, valueToPut);
        }

    }

    @Override
    public JvmtiThreadInfoWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiThreadInfoWrapper(jni, pointer);
    }

}

