/* Copyright (c) 2023 Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.oracle.truffle.espresso.vm.structs;

import java.nio.ByteBuffer;

import com.oracle.truffle.api.interop.TruffleObject;

import com.oracle.truffle.espresso.runtime.staticobject.StaticObject;
import com.oracle.truffle.espresso.jni.JniEnv;
import com.oracle.truffle.espresso.ffi.RawPointer;

@SuppressWarnings(value = "unused") public final class JvmtiStackInfo extends StructStorage<JvmtiStackInfo.JvmtiStackInfoWrapper> {

    final int thread;

    final int state;

    final int frame_buffer;

    final int frame_count;

    JvmtiStackInfo(MemberOffsetGetter offsetGetter) {
        super(offsetGetter.getInfo("_jvmtiStackInfo"));
        thread = (int) offsetGetter.getOffset("_jvmtiStackInfo", "thread");
        state = (int) offsetGetter.getOffset("_jvmtiStackInfo", "state");
        frame_buffer = (int) offsetGetter.getOffset("_jvmtiStackInfo", "frame_buffer");
        frame_count = (int) offsetGetter.getOffset("_jvmtiStackInfo", "frame_count");
    }

    public final class JvmtiStackInfoWrapper extends StructWrapper {

        JvmtiStackInfoWrapper(JniEnv jni, TruffleObject pointer) {
            super(jni, pointer, structSize);
        }

        public StaticObject thread() {
            return getObject(thread);
        }

        public void thread(StaticObject valueToPut) {
            putObject(thread, valueToPut);
        }

        public int state() {
            return getInt(state);
        }

        public void state(int valueToPut) {
            putInt(state, valueToPut);
        }

        public TruffleObject frameBuffer() {
            return getPointer(frame_buffer);
        }

        public void frameBuffer(TruffleObject valueToPut) {
            putPointer(frame_buffer, valueToPut);
        }

        public int frameCount() {
            return getInt(frame_count);
        }

        public void frameCount(int valueToPut) {
            putInt(frame_count, valueToPut);
        }

    }

    @Override
    public JvmtiStackInfoWrapper wrap(JniEnv jni, TruffleObject pointer) {
        return new JvmtiStackInfoWrapper(jni, pointer);
    }

}

