/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.persistence;

import java.time.Instant;
import java.util.List;
import java.util.Optional;

import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.tokenauth.model.ApiToken;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;

/**
 * {@link ApiToken} persistence.
 */
public interface ApiTokenPersistence extends JpaRepository<ApiToken, Long> {

	/**
	 * Get {@link ApiToken} by {@code token} value.
	 *
	 * @param token The token
	 */
	Optional<ApiToken> findByToken(String token);

	/**
	 * List API tokens by sid.
	 *
	 * @param sid The sid
	 * @return the list of tokens
	 */
	List<ApiToken> findAllBySid(AclSid sid);

	/**
	 * List API tokens with expiration date before the cut-off date.
	 *
	 * @param cutoffDate The cut-off time
	 * @return the list of tokens with expiration date earlier than {@code before}
	 */
	@Query("select apitoken from ApiToken apitoken where apitoken.expires > :cutoffDate")
	List<AclSid> listExpiredApiTokens(Instant cutoffDate);

}
