/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.spring;

import java.util.Collection;

import org.genesys.blocks.oauth.model.OAuthClient;
import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.tokenauth.model.ApiToken;
import org.springframework.security.core.CredentialsContainer;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.AuthenticationUserDetailsService;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

/**
 * Load details based on token.
 */
public interface ApiTokenDetailsService extends AuthenticationUserDetailsService<ApiTokenAuthenticationToken> {

	/**
	 * @param token The pre-authenticated authentication token
	 * @return UserDetails for the given authentication token, never null.
	 * @throws UsernameNotFoundException if no user details can be found for the given
	 * authentication token
	 */
	@Override
	ApiTokenDetails<? extends AclSid> loadUserDetails(ApiTokenAuthenticationToken token) throws UsernameNotFoundException;

	/**
	 * AclSid details for API Token authenticaton.
	 */
	abstract class ApiTokenDetails<T extends AclSid> implements UserDetails, CredentialsContainer {
		protected T sid;
		private ApiToken token;

		/**
		 * Make details for sid and token.
		 *
		 * @param sid The authenticated AclSid
		 * @param token The token
		 */
		public ApiTokenDetails(T sid, ApiToken token) {
			this.sid = sid;
			this.token = token;
		}

		@Override
		public final String getPassword() {
			return token.getToken();
		}

		@Override
		public final String getUsername() {
			return sid.getSid();
		}

		@Override
		public boolean isAccountNonExpired() {
			return sid.isActive();
		}

		@Override
		public boolean isAccountNonLocked() {
			return sid.isActive();
		}

		@Override
		public final boolean isCredentialsNonExpired() {
			return token.isCredentialsNonExpired();
		}

		@Override
		public final boolean isEnabled() {
			return sid.isActive() && token.isActive();
		}

		@Override
		public void eraseCredentials() {
			this.token = null;
		}
	}

	/**
	 * ApiTokenDetails for an OAuth client.
	 */
	final class ApiTokenClientDetails extends ApiTokenDetails<OAuthClient> {

		public ApiTokenClientDetails(OAuthClient sid, ApiToken token) {
			super(sid, token);
		}

		@Override
		public Collection<? extends GrantedAuthority> getAuthorities() {
			return this.sid.getAuthorities();
		}

	}

	/**
	 * ApiTokenDetails implementation for a user account.
	 */
	final class ApiTokenUserDetails extends ApiTokenDetails<BasicUser<?>> {
		
		public ApiTokenUserDetails(BasicUser<?> sid, ApiToken token) {
			super(sid, token);
		}

		@Override
		public Collection<? extends GrantedAuthority> getAuthorities() {
			return this.sid.getAuthorities();
		}

		@Override
		public boolean isAccountNonExpired() {
			return sid.isAccountNonExpired();
		}

		@Override
		public boolean isAccountNonLocked() {
			return sid.isAccountNonLocked();
		}
	}
}
