/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.spring;

import java.util.Collection;

import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.core.GrantedAuthority;

/**
 * {@link org.springframework.security.core.Authentication} implementation for
 * API Token authentication.
 *
 * @author Matija Obreza
 */
public class ApiTokenAuthenticationToken extends AbstractAuthenticationToken {

  private final Object principal;

  private Object credentials;

  /**
   * Constructor used for an authentication request. The
   * {@link org.springframework.security.core.Authentication#isAuthenticated()}
   * will return <code>false</code>. The principal is unknown at this stage.
   * 
   * @param token The API token
   */
  public ApiTokenAuthenticationToken(String token) {
    super(null);
    this.principal = null;
    this.credentials = token;
  }

  /**
   * Constructor used for an authentication response. The
   * {@link org.springframework.security.core.Authentication#isAuthenticated()}
   * will return <code>true</code>.
   * 
   * @param aPrincipal    The authenticated principal
   * @param anAuthorities The granted authorities
   */
  public ApiTokenAuthenticationToken(Object aPrincipal, Object aCredentials, Collection<? extends GrantedAuthority> anAuthorities) {
    super(anAuthorities);
    this.principal = aPrincipal;
    this.credentials = aCredentials;
    setAuthenticated(true);
  }

  /**
   * Get the credentials
   */
  @Override
  public Object getCredentials() {
    return this.credentials;
  }

  /**
   * Get the principal
   */
  @Override
  public Object getPrincipal() {
    return this.principal;
  }

  @Override
  public void eraseCredentials() {
    this.credentials = null;
  }
}