/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.spring;

import org.genesys.blocks.tokenauth.spring.ApiTokenDetailsService.ApiTokenDetails;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.Ordered;
import org.springframework.security.authentication.AccountStatusUserDetailsChecker;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.util.Assert;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * API Token authentication provider
 */
@Slf4j
@Getter
@Setter
public class ApiTokenAuthenticationProvider implements AuthenticationProvider, InitializingBean, Ordered {
	private int order = -1; // default: same as non-ordered

	private ApiTokenDetailsService apiTokenDetailsService;

	private UserDetailsChecker userDetailsChecker = new AccountStatusUserDetailsChecker();

	@Override
	public void afterPropertiesSet() throws Exception {
		Assert.notNull(this.apiTokenDetailsService, "An ApiTokenDetailsService must be set");
	}

	/**
	 * Authenticate the given PreAuthenticatedAuthenticationToken.
	 * <p>
	 * If the principal contained in the authentication object is null, the request will
	 * be ignored to allow other providers to authenticate it.
	 */
	@Override
	public Authentication authenticate(Authentication authentication) throws AuthenticationException {
		if (!supports(authentication.getClass())) {
			return null;
		}
		log.debug("API-Token authentication: {}", authentication);
		// if (authentication.getPrincipal() == null) {
		// 	log.debug("No pre-authenticated principal found in request.");
		// 	if (this.throwExceptionWhenTokenRejected) {
		// 		throw new BadCredentialsException("No pre-authenticated principal found in request.");
		// 	}
		// 	return null;
		// }
		if (authentication.getCredentials() == null) {
			log.debug("No API-Token credentials found in request.");
			throw new BadCredentialsException("No token found in request.");
		}
		ApiTokenDetails<?> userDetails = this.apiTokenDetailsService.loadUserDetails((ApiTokenAuthenticationToken) authentication);
		this.userDetailsChecker.check(userDetails);
		ApiTokenAuthenticationToken result = new ApiTokenAuthenticationToken(userDetails.sid, authentication.getCredentials(), userDetails.getAuthorities());
		result.setDetails(userDetails);
		return result;
	}

	/**
	 * Indicate that this provider only supports PreAuthenticatedAuthenticationToken
	 * (sub)classes.
	 */
	@Override
	public final boolean supports(Class<?> authentication) {
		return ApiTokenAuthenticationToken.class.isAssignableFrom(authentication);
	}

	@Override
	public int getOrder() {
		return this.order;
	}

	public void setOrder(int i) {
		this.order = i;
	}

}
