/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.service;

import org.genesys.blocks.security.filter.InvalidatedTokenFilter;
import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.tokenauth.model.ApiToken;
import org.genesys.blocks.tokenauth.spring.ApiTokenDetailsService;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;

import java.time.Instant;
import java.util.List;

/**
 * API Token services
 */
public interface ApiTokenService {

	/**
	 * Generate the encoded token string.
	 * 
	 * @param token token value
	 * @return encoded token value
	 */
	String encodeToken(String token);

	/**
	 * Register a new API token for the specified {@code sid} with a label and
	 * expiration date.
	 * 
	 * @param sid     the SID represented by the token
	 * @param label   a token friendly name, unique to the {@code sid}
	 * @param expires the expiration date of the token
	 * @return a new API token with {@code token.token} provided in plain-text
	 */
	ApiToken createToken(AclSid sid, String label, Instant expires);

	ApiToken loadById(Long id);

	/**
	 * Get token by its persisted value.
	 * This is used by the {@link InvalidatedTokenFilter} and {@link ApiTokenDetailsService}.
	 * 
	 * @param encodedToken encoded token
	 * @return {@code null} if token not found
	 */
	ApiToken getToken(String encodedToken);

	List<ApiToken> listTokensForSid(AclSid sid);

	Page<ApiToken> listTokens(Pageable page);

	ApiToken remove(ApiToken apiToken);

	ApiToken update(ApiToken apiToken);
}
