/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.tokenauth.model;

import java.time.Instant;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Index;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.Table;

import com.fasterxml.jackson.annotation.JsonView;
import org.genesys.blocks.annotations.NotCopyable;
import org.genesys.blocks.model.AuditedVersionedModel;
import org.genesys.blocks.model.Copyable;
import org.genesys.blocks.model.JsonViews;
import org.genesys.blocks.security.model.AclSid;

import lombok.Getter;
import lombok.Setter;

/**
 * {@code ApiToken} supports token-based authentication. Each token is unique in the system and
 * links to one and only one {@link AclSid}.
 */
@Entity
@Table(name = "api_token", indexes = {
	@Index(name = "apitoken_token", columnList = "token", unique = true), // Unique index on token
	@Index(name = "apitoken_sid", columnList = "sid"), // Index on sid
	@Index(name = "apitoken_expires", columnList = "expires DESC"), // Index on expires
	@Index(name = "apitoken_sid_label", columnList = "sid, label", unique = true), // Index on sid and label
})
@Getter
@Setter
public class ApiToken extends AuditedVersionedModel implements Copyable<ApiToken> {

	/** The token */
	@Column(length = 128, unique = true, nullable = false, updatable = false)
	@JsonView({ JsonViews.Internal.class })
	@NotCopyable
	private String token;

	/** Token label helps users identify the purpose of the token */
	@Column(length = 64)
	private String label;

	/** Token expiration date */
	@Column(nullable = true, updatable = true)
	private Instant expires;

	/** Token owner */
	@ManyToOne(fetch = FetchType.EAGER, cascade = {})
	@JoinColumn(name = "sid", updatable = false, nullable = false)
	@NotCopyable
	private AclSid sid;

	@Override
	public boolean canEqual(Object other) {
		return other instanceof ApiToken;
	}

	/**
	 * Check if token <b>is not</b> expired.
	 * API Token is non-expired when {@code expires} is null or is in the future.
	 * Token is expired if {@code expires} if non-null and it is in the past.
	 *
	 * @return {@code true} if expired.
	 */
	public boolean isCredentialsNonExpired() {
		return expires == null || Instant.now().isBefore(expires);
	}
}
