/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.model;

import java.util.List;

import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorType;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.OneToMany;
import javax.persistence.Table;

import org.genesys.blocks.model.AuditedVersionedModel;
import org.hibernate.annotations.DiscriminatorOptions;

import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonIdentityReference;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;

import lombok.Getter;
import lombok.Setter;

/**
 * ACL SID uniquely identifies any principal or authority in the system ("SID"
 * stands for "security identity").
 */
@Entity
@Table(name = "acl_sid")

/// User and OAuthClient are both SID entities and we need a way to pull them
/// together. Using AclSid seems like the best approach.
/// JOINED inheritance must be used because we need to be able to query acl_sid
/// table with JDBC.
@Inheritance(strategy = InheritanceType.JOINED)
@DiscriminatorColumn(name = "type", discriminatorType = DiscriminatorType.INTEGER)
@DiscriminatorValue(value = "0")
@DiscriminatorOptions(force = false)
@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "sid")
@Cacheable
@Getter
@Setter
public class AclSid extends AuditedVersionedModel {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -8665345718313672678L;

	public static final String OIDC_SID_ID = "sidId";

	/** The principal. */
	@Column(name = "principal", nullable = false, length = 1)
	private boolean principal;

	/** The sid. */
	@JsonIdentityReference(alwaysAsId = true)
	@Column(name = "sid", nullable = false, length = 100)
	private String sid;

	/** The object identities. */
	@JsonIgnore
	// no cascade delete, those object identities must stay because other ACL
	// entries rely on them
	@OneToMany(mappedBy = "ownerSid", fetch = FetchType.LAZY, cascade = {})
	private List<AclObjectIdentity> objectIdentities;

	/** The acl entries. */
	@JsonIgnore
	@OneToMany(mappedBy = "aclSid", fetch = FetchType.LAZY, cascade = CascadeType.REMOVE)
	private List<AclEntry> aclEntries;

	/**
	 * Subclasses should override this method and return a sensible display name for
	 * the SID.
	 *
	 * @return SID full name
	 */
	public String getFullName() {
		return this.sid;
	}

	@Override
	public boolean canEqual(Object other) {
		return other instanceof AclSid;
	}

	@Override
	public String toString() {
		return this.sid;
	}
}
