/*
 * Copyright 2023 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.lockout;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.service.BasicUserService;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;

import lombok.extern.slf4j.Slf4j;

/**
 * Register this on your form logins.
 *
 * @author ahrybeniuk
 */
@Slf4j
public class LoginSuccessHandler extends SavedRequestAwareAuthenticationSuccessHandler {

	private final AccountLockoutManager lockoutManager;

	/** The user service. */
	private BasicUserService<?, ?> userService;

	public LoginSuccessHandler(AccountLockoutManager lockoutManager, BasicUserService<?, ?> userService) {
		this.lockoutManager = lockoutManager;
		this.userService = userService;
	}

	@Override
	public void onAuthenticationSuccess(HttpServletRequest request, HttpServletResponse response,
		Authentication authentication) throws ServletException, IOException {
		super.onAuthenticationSuccess(request, response, authentication);
		final Object principal = authentication.getPrincipal();
		log.warn("Authentication successful: {}", principal);

		if (principal instanceof BasicUser<?>) {
			String userName = ((BasicUser<?>) principal).getUsername();
			lockoutManager.handleSuccessfulLogin(userName);
			try {
				userService.updateLastLogin(userName);
			} catch (NoUserFoundException e) {
				// Ignore
			}
		}
	}
}
