/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.genesys.blocks.util;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * Based on this <a href=
 * "https://stackoverflow.com/questions/28005761/how-to-access-a-method-a-spring-bean-from-a-class-not-in-the-spring-container/28005923#28005923">Stackoveflow
 * comment</a>.
 * 
 * The bean exposes the current Spring application context to static methods.
 * Register it as
 * 
 * <pre>
 * &#64;Bean
 * public CurrentApplicationContext currentAppContext() {
 * 	return new CurrentApplicationContext();
 * }
 * </pre>
 * 
 * Once registered, static methods can obtain bean references:
 * 
 * <pre>
 * static {
 * 	SomeService myService = CurrentApplicationContext.getContext().getBean(...);
 * }
 * 
 * public static void foo() {
 * 	SomeBean bean = CurrentApplicationComntext.getContext().getBean(...);
 * }
 * </pre>
 */
public class CurrentApplicationContext implements ApplicationContextAware {

	/** The application context. */
	static volatile ApplicationContext applicationContext = null;

	public void setApplicationContext(ApplicationContext context) throws BeansException {
		synchronized (CurrentApplicationContext.class) {
			applicationContext = context;
		}
	}

	/**
	 * Note that this is a static method which expose ApplicationContext.
	 *
	 * @return the instance of the current Spring {@link ApplicationContext}
	 */
	public static ApplicationContext getContext() {
		return applicationContext;
	}

}