/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.serialization;

import static org.springframework.security.acls.domain.BasePermission.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclSid;

/**
 * Simple POJO for SID's permissions.
 */
public class SidPermissions extends Permissions {

	/** SID having these permisions. */
	public AclSid sid;

	/**
	 * Set ACL SID.
	 *
	 * @param aclSid the acl sid
	 * @return the sid permissions
	 */
	public SidPermissions aclSid(AclSid aclSid) {
		this.sid = aclSid;
		return this;
	}
	
	/**
	 * Get SID.
	 *
	 * @return SID of current permissions set
	 */
	public AclSid getSid() {
		return sid;
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.serialization.Permissions#toString()
	 */
	@Override
	public String toString() {
		return "sid=" + sid + " p=" + super.toString();
	}

	/**
	 * Convert List of {@link AclEntry} to List of {@link SidPermissions}.
	 *
	 * @param aclEntries the acl entries
	 * @return the list
	 */
	public static List<SidPermissions> fromEntries(List<AclEntry> aclEntries) {
		if (aclEntries == null) {
			return null;
		}
		if (aclEntries.isEmpty()) {
			return Collections.emptyList();
		}

		List<SidPermissions> converted = new ArrayList<>(1 + aclEntries.size() / 5);

		for (final AclEntry entry : aclEntries) {
			// find existing entry
			SidPermissions permission = converted.stream().filter(p -> entry.getAclSid().getId().equals(p.sid.getId())).findFirst().orElse(null);

			// or make one
			if (permission == null) {
				permission = new SidPermissions().aclSid(entry.getAclSid());
				converted.add(permission);
			}

			if (CREATE.getMask() == entry.getMask()) {
				permission.create = entry.isGranting();
			} else if (READ.getMask() == entry.getMask()) {
				permission.read = entry.isGranting();
			} else if (WRITE.getMask() == entry.getMask()) {
				permission.write = entry.isGranting();
			} else if (DELETE.getMask() == entry.getMask()) {
				permission.delete = entry.isGranting();
			} else if (ADMINISTRATION.getMask() == entry.getMask()) {
				permission.manage = entry.isGranting();
			}
		}

		return converted;
	}
}
