/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.model;

import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;

import org.genesys.blocks.model.BasicModel;
import org.genesys.blocks.model.JsonViews;
import org.genesys.blocks.security.serialization.AclEntriesToPermissions;

import com.fasterxml.jackson.annotation.JsonIdentityInfo;
import com.fasterxml.jackson.annotation.JsonIdentityReference;
import com.fasterxml.jackson.annotation.JsonView;
import com.fasterxml.jackson.annotation.ObjectIdGenerators;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

/**
 * ACL Object Identity represents a specific ACL-aware entity (combination of
 * {@link AclClass} and {#link objectIdIdentity}).
 *
 * It records the owner of the entity and entity's parent object if any.
 */
@Entity
@Table(name = "acl_object_identity", uniqueConstraints = @UniqueConstraint(columnNames = { "object_id_class", "object_id_identity" }))
@JsonIdentityInfo(generator = ObjectIdGenerators.PropertyGenerator.class, property = "id")
@Getter
@Setter
@ToString(callSuper = true)
public class AclObjectIdentity extends BasicModel {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -8987281958564243010L;

	/** The acl class. */
	@ManyToOne(fetch = FetchType.EAGER, cascade = CascadeType.PERSIST)
	@JoinColumn(name = "object_id_class", nullable = false)
	private AclClass aclClass;

	/** The parent object. */
	@ManyToOne(fetch = FetchType.EAGER, cascade = CascadeType.PERSIST)
	@JoinColumn(name = "parent_object", nullable = true)
	@JsonIdentityReference(alwaysAsId = true)
	@ToString.Exclude
	private AclObjectIdentity parentObject;

	/** The owner sid. */
	@ManyToOne(fetch = FetchType.EAGER, cascade = {})
	@JoinColumn(name = "owner_sid", nullable = true)
	@JsonIdentityReference(alwaysAsId = false)
	@JsonView(JsonViews.Minimal.class)
	private AclSid ownerSid;

	/** The object id identity. */
	@Column(name = "object_id_identity", nullable = false)
	private long objectIdIdentity;

	/** The entries inheriting. */
	@Column(name = "entries_inheriting", nullable = false, length = 1)
	private boolean entriesInheriting;

	/** The acl entries. */
	// @JsonIgnore
	@JsonSerialize(converter = AclEntriesToPermissions.class)
	@OneToMany(mappedBy = "aclObjectIdentity", fetch = FetchType.LAZY, cascade = CascadeType.REMOVE, orphanRemoval = true)
	private List<AclEntry> aclEntries;

	@Override
	public boolean canEqual(Object other) {
		return other instanceof AclObjectIdentity;
	}
}
