/*
 * Copyright 2019 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.model.filters;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.deser.std.UUIDDeserializer;
import com.querydsl.core.types.Predicate;
import org.apache.commons.collections4.CollectionUtils;
import org.genesys.blocks.model.AuditedVersionedModel;
import org.genesys.blocks.model.QUuidModel;
import org.genesys.blocks.model.UuidModel;

import com.querydsl.core.types.dsl.EntityPathBase;

/**
 * {@link UuidModel} match by sample filters.
 *
 * @param <T> the generic type
 * @param <R> the generic type
 */
public abstract class UuidModelFilter<T extends UuidModelFilter<T, R>, R extends AuditedVersionedModel> extends AuditedVersionedModelFilter<T, R> {

	private static final long serialVersionUID = -2866906205548608641L;

	/** The created by. */
	@JsonDeserialize(contentUsing = UUIDDeserializer.class)
	public Set<UUID> uuid;

	/**
	 * Collects list of filter predicates
	 *
	 * @param instance the instance of Q-type of <em>R</em>
	 * @param uuidModel the uuid model
	 * @return list of predicates
	 */
	protected List<Predicate> collectPredicates(final EntityPathBase<R> instance, final QUuidModel uuidModel) {
		List<Predicate> predicates = super.collectPredicates(instance, uuidModel._super);
		if (CollectionUtils.isNotEmpty(uuid)) {
			predicates.add(uuidModel.uuid.in(uuid));
		}
		return predicates;
	}

	/**
	 * Uuid.
	 *
	 * @return the sets the
	 */
	public synchronized Set<UUID> uuid() {
		if (uuid == null) {
			uuid = new HashSet<>();
		}
		return uuid;
	}
}
