/*
 * Copyright 2019 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.model.filters;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.querydsl.core.types.Predicate;
import org.apache.commons.collections4.CollectionUtils;
import org.genesys.blocks.model.AuditedVersionedModel;
import org.genesys.blocks.model.QAuditedVersionedModel;

import com.querydsl.core.types.dsl.EntityPathBase;

/**
 * {@link AuditedVersionedModel} match by sample filters.
 *
 * @param <T> the generic type
 * @param <R> the generic type
 */
public abstract class AuditedVersionedModelFilter<T extends AuditedVersionedModelFilter<T, R>, R extends AuditedVersionedModel> extends VersionedModelFilter<T, R> {

	private static final long serialVersionUID = 4488044011334369267L;

	/** The created by. */
	public Set<Long> createdBy;

	/** The last modified by. */
	public Set<Long> lastModifiedBy;

	/** The created date. */
	public InstantFilter createdDate;

	/** The last modified date. */
	public InstantFilter lastModifiedDate;

	/**
	 * Collects list of filter predicates
	 *
	 * @param instance the instance of Q-type of <em>R</em>
	 * @param auditedVersionedModel the audited versioned model
	 * @return list of predicates
	 */
	protected List<Predicate> collectPredicates(final EntityPathBase<R> instance, final QAuditedVersionedModel auditedVersionedModel) {
		List<Predicate> predicates = super.collectPredicates(instance, auditedVersionedModel._super);
		if (CollectionUtils.isNotEmpty(createdBy)) {
			predicates.add(auditedVersionedModel.createdBy.in(createdBy));
		}
		if (CollectionUtils.isNotEmpty(lastModifiedBy)) {
			predicates.add(auditedVersionedModel.lastModifiedBy.in(lastModifiedBy));
		}
		if (createdDate != null) {
			predicates.add(createdDate.buildQuery(auditedVersionedModel.createdDate));
		}
		if (lastModifiedDate != null) {
			predicates.add(lastModifiedDate.buildQuery(auditedVersionedModel.lastModifiedDate));
		}
		return predicates;
	}

	/**
	 * Created by.
	 *
	 * @return the sets the
	 */
	public synchronized Set<Long> createdBy() {
		if (createdBy == null) {
			createdBy = new HashSet<>();
		}
		return createdBy;
	}

	/**
	 * Last modified by.
	 *
	 * @return the sets the
	 */
	public synchronized Set<Long> lastModifiedBy() {
		if (lastModifiedBy == null) {
			lastModifiedBy = new HashSet<>();
		}
		return lastModifiedBy;
	}

	/**
	 * Created date.
	 *
	 * @return the date filter
	 */
	public synchronized InstantFilter createdDate() {
		if (createdDate == null) {
			createdDate = new InstantFilter();
		}
		return createdDate;
	}

	/**
	 * Last modified date.
	 *
	 * @return the date filter
	 */
	public synchronized InstantFilter lastModifiedDate() {
		if (lastModifiedDate == null) {
			lastModifiedDate = new InstantFilter();
		}
		return lastModifiedDate;
	}
}
