/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.component;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.genesys.blocks.oauth.model.OAuthClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.provider.ClientDetailsService;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.web.filter.OncePerRequestFilter;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

/**
 * Filter OAuth2 requests and check that Origin header matches what we have on
 * file.
 */
public class OAuthClientOriginCheckFilter extends OncePerRequestFilter {

	@Autowired
	@Qualifier("oauthService")
	private ClientDetailsService clientDetailsService;

	private LoadingCache<String, Set<String>> clientOriginsCache = CacheBuilder.newBuilder().maximumSize(100).expireAfterWrite(10, TimeUnit.MINUTES).build(
		new CacheLoader<String, Set<String>>() {
			public Set<String> load(String clientId) {
				if (logger.isInfoEnabled()) {
					logger.info("Loading allowed origins for client: " + clientId);
				}
				OAuthClient clientDetails = (OAuthClient) clientDetailsService.loadClientByClientId(clientId);
				return clientDetails.getAllowedOrigins();
			}
		});

	@Override
	protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
		Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		if (authentication != null && authentication instanceof OAuth2Authentication) {
			if (!checkValidOrigin(request, (OAuth2Authentication) authentication)) {
				response.sendError(403, "Request origin not valid");
				return;
			}
		} else {
			if (logger.isDebugEnabled()) {
				logger.debug("Authentication null for origin: " + request.getHeader("Origin"));
			}
		}
		filterChain.doFilter(request, response);
	}

	private boolean checkValidOrigin(HttpServletRequest request, OAuth2Authentication authAuth) {
		if (logger.isTraceEnabled()) {
			logger.trace(request.getRequestURI());
			for (String headerName : Collections.list(request.getHeaderNames())) {
				logger.trace(">> " + headerName + ": " + request.getHeader(headerName));
			}
		}
		String reqOrigin = request.getHeader("Origin");
		String reqReferrer = request.getHeader("Referer"); // GET requests don't carry Origin?

		if (authAuth.getOAuth2Request() != null) {
			boolean isGet = request.getMethod().equalsIgnoreCase("get");
			String clientId = authAuth.getOAuth2Request().getClientId();
			
			try {
				Set<String> allowedOrigins = clientOriginsCache.get(clientId);
				
				if (!allowedOrigins.isEmpty()) {
					if (reqOrigin == null && reqReferrer == null) {
						if (logger.isInfoEnabled()) {
							logger.info("No origin/referrer header in request. Denying.");
						}
						return false;
					}
					for (String allowedOrigin : allowedOrigins) {
						if (reqOrigin != null && reqOrigin.startsWith(allowedOrigin)) {
							if (logger.isDebugEnabled()) {
								logger.debug("Origin match: " + reqOrigin + " for " + allowedOrigin);
							}
							return true;
						}

						if ((isGet || reqOrigin == null) && reqReferrer != null && reqReferrer.startsWith(allowedOrigin)) {
							if (logger.isDebugEnabled()) {
								logger.debug("Referrer match: " + reqReferrer + " for " + allowedOrigin);
							}
							return true;
						}
					}
					// No declared origins match
					if (logger.isInfoEnabled()) {
						logger.info("No origin/referrer match: " + reqOrigin + " or " + reqReferrer + " in " + allowedOrigins.toString());
					}
					return false;
				} else {
					if (reqOrigin != null || reqReferrer != null) {
						if (logger.isInfoEnabled()) {
							logger.info(clientId + " may not be used from browsers. Denying.");
						}
						return false;
					}
					return true;
				}
			} catch (ExecutionException e) {
				logger.warn("Error loading client origins", e);
			}
		}
		if (logger.isDebugEnabled()) {
			logger.debug("Allowing request with Origin: " + reqOrigin);
		}
		return true;
	}
}