/*
 * Copyright 2019 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.service;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.persistence.EntityNotFoundException;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.genesys.blocks.oauth.model.OAuthClient;
import org.genesys.blocks.oauth.model.OAuthRole;
import org.genesys.blocks.oauth.model.QOAuthClient;
import org.genesys.blocks.oauth.persistence.OAuthClientRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.ClientRegistrationException;
import org.springframework.security.oauth2.provider.NoSuchClientException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.querydsl.core.types.Predicate;

/**
 * The Class OAuthServiceImpl.
 */
@Service
@Transactional(readOnly = true)
public class OAuthServiceImpl implements OAuthClientDetailsService {

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(OAuthServiceImpl.class);

	/** The hostname. */
	@Value("${host.name}")
	private String hostname;

	/** The oauth client repository. */
	@Autowired
	private OAuthClientRepository oauthClientRepository;

	/** The password encoder. */
	@Autowired
	public PasswordEncoder passwordEncoder;

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetailsService#
	 * loadClientByClientId(java.lang.String)
	 */
	@Override
	@Cacheable(cacheNames = { "oauthclient" }, key = "#clientId", unless = "#result == null")
	public ClientDetails loadClientByClientId(final String clientId) throws ClientRegistrationException {
		final OAuthClient client = oauthClientRepository.findByClientId(clientId);
		if (client == null) {
			throw new NoSuchClientException(clientId);
		}
		return lazyLoad(client);
	}

	private OAuthClient lazyLoad(OAuthClient client) {
		if (client != null) {
			client.getRoles().size();
			client.getRoles().add(OAuthRole.EVERYONE);
		}
		return client;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.oauth.service.OAuthClientDetailsService#listClientDetails(
	 * )
	 */
	@Override
	public List<OAuthClient> listClientDetails() {
		return oauthClientRepository.findAll(Sort.by("clientId"));
	}

	@Override
	public Page<OAuthClient> listClientDetails(Pageable pageable) {
		return oauthClientRepository.findAll(pageable);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.oauth.service.OAuthClientDetailsService#getClient(java.
	 * lang.String)
	 */
	@Override
	public OAuthClient getClient(final String clientId) {
		OAuthClient client = oauthClientRepository.findByClientId(clientId);
		if (client != null)
			client.getRoles().size();
		return client;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.oauth.service.OAuthClientDetailsService#removeClient(org.
	 * genesys.blocks.oauth.model.OAuthClient)
	 */
	@Override
	@Transactional
	
	@CacheEvict(cacheNames = { "oauthclient" }, key = "#client.clientId", condition = "#client != null")
	public OAuthClient removeClient(final OAuthClient client) {
		oauthClientRepository.delete(client);
		return client;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.oauth.service.OAuthClientDetailsService#addClient(org.
	 * genesys.blocks.oauth.model.OAuthClient)
	 */
	@Override
	@Transactional
	public OAuthClient addClient(OAuthClient client) {
		final String clientId = RandomStringUtils.randomAlphanumeric(5).toLowerCase() + "." + RandomStringUtils.randomAlphanumeric(20).toLowerCase() + "@" + hostname;
		final String clientSecret = RandomStringUtils.randomAlphanumeric(32);

		final OAuthClient newClient = new OAuthClient();
		newClient.apply(client);
		newClient.setClientId(clientId);
		newClient.setClientSecret(passwordEncoder.encode(clientSecret));

		return lazyLoad(oauthClientRepository.save(newClient));
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.oauth.service.OAuthClientDetailsService#updateClient(long,
	 * int, org.genesys.blocks.oauth.model.OAuthClient)
	 */
	@Override
	@Transactional
	@CacheEvict(cacheNames = { "oauthclient" }, key = "#updates.clientId", condition = "#updates != null")
	public OAuthClient updateClient(final long id, final int version, final OAuthClient updates) {
		OAuthClient client = oauthClientRepository.findByIdAndVersion(id, version);
		client.apply(updates);
		return lazyLoad(oauthClientRepository.save(client));
	}

	@Override
	public List<OAuthClient> autocompleteClients(final String term, int limit) {
		if (StringUtils.isBlank(term) || term.length() < 1)
			return Collections.emptyList();

		LOG.debug("Autocomplete for={}", term);

		Predicate predicate = QOAuthClient.oAuthClient.title.startsWithIgnoreCase(term)
			// clientId
			.or(QOAuthClient.oAuthClient.clientId.startsWithIgnoreCase(term))
			// description contains
			.or(QOAuthClient.oAuthClient.description.contains(term));

		return oauthClientRepository.findAll(predicate, PageRequest.of(0, Math.min(100, limit), Sort.by("title"))).getContent();
	}

	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#oauthClient, 'ADMINISTRATION')")
	public final String resetSecret(OAuthClient oauthClient) {
		assert oauthClient != null;
		assert oauthClient.getId() != null;

		oauthClient = oauthClientRepository.findById(oauthClient.getId()).orElseThrow(() -> new EntityNotFoundException("Record not found."));

		String oldHash = oauthClient.getClientSecret();
		String newHash = null;
		String clientSecret = null;
		do {
			clientSecret = RandomStringUtils.randomAlphanumeric(32);
			newHash = passwordEncoder.encode(clientSecret);
		} while (oldHash != null && oldHash.equals(newHash));

		oauthClient.setClientSecret(newHash);
		oauthClientRepository.save(oauthClient);
		return clientSecret;
	}

	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#oauthClient, 'ADMINISTRATION')")
	public final OAuthClient removeSecret(OAuthClient oauthClient) {
		assert oauthClient != null;
		assert oauthClient.getId() != null;

		oauthClient = oauthClientRepository.findById(oauthClient.getId()).orElseThrow(() -> new EntityNotFoundException("Record not found."));
		if (oauthClient.getAuthorizedGrantTypes().contains("client_credentials")) {
			throw new RuntimeException("OAuth Client with client_credentials grant must have a secret");
		}
		oauthClient.setClientSecret(null);
		oauthClient = oauthClientRepository.save(oauthClient);
		return lazyLoad(oauthClient);
	}

	@Override
	public boolean isOriginRegistered(String origin) {
		AtomicBoolean found = new AtomicBoolean(false);

		oauthClientRepository.findAll(QOAuthClient.oAuthClient.origins.contains(origin)).forEach(client -> {
			if (client.getAllowedOrigins().contains(origin)) {
				found.set(true);
			}
		});

		return found.get();
	}
}
