/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.serialization;

import org.genesys.blocks.security.SecurityContextUtil;
import org.genesys.blocks.security.model.AclAwareModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.acls.domain.BasePermission;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.cfg.MapperConfig;
import com.fasterxml.jackson.databind.introspect.AnnotatedClass;
import com.fasterxml.jackson.databind.introspect.BeanPropertyDefinition;
import com.fasterxml.jackson.databind.ser.VirtualBeanPropertyWriter;
import com.fasterxml.jackson.databind.util.Annotations;

/**
 * The CurrentPermissionsWriter is applied to {@link AclAwareModel} and it
 * instructs Jackson to to include {@link Permissions} for current SID for every
 * ACL aware entity.
 * 
 * Serialization is enabled <code>@JsonAppend</code> annotation on
 * <code>AclAwareModel</code>:
 * 
 * <pre>
 *  &#64;JsonAppend(props = { @JsonAppend.Prop(name="_permissions", value = CurrentPermissionsWriter.class, type=CurrentPermissions.class) })
 *  public interface AclAwareModel...
 * </pre>
 * 
 * To be able to access the current permissions of the current SID, this code
 * relies on an instance of
 * <code>org.genesys.blocks.util.CurrentApplicationContext</code> to be
 * registered in the Spring application context.
 * 
 */
public class CurrentPermissionsWriter extends VirtualBeanPropertyWriter {

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(CurrentPermissionsWriter.class);

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = 1L;


	private static final Permissions NO_PERMISSIONS = new Permissions().grantNone();

	/**
	 * Instantiates a new current permissions writer.
	 */
	public CurrentPermissionsWriter() {
		LOG.trace("CurrentPermissionsWriter");
	}

	/**
	 * Instantiates a new current permissions writer.
	 *
	 * @param propDef the prop def
	 * @param annotations the annotations
	 * @param type the type
	 */
	public CurrentPermissionsWriter(BeanPropertyDefinition propDef, Annotations annotations, JavaType type) {
		super(propDef, annotations, type);
		LOG.trace("CurrentPermissionsWriter");
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * com.fasterxml.jackson.databind.ser.VirtualBeanPropertyWriter#value(java.lang.
	 * Object, com.fasterxml.jackson.core.JsonGenerator,
	 * com.fasterxml.jackson.databind.SerializerProvider)
	 */
	@Override
	protected Object value(Object bean, JsonGenerator gen, SerializerProvider prov) throws Exception {
		if (bean == null || !(bean instanceof AclAwareModel)) {
			// Skip nulls
			return null;
		}
		
		AclAwareModel aclAwareModel = (AclAwareModel) bean;

		if (aclAwareModel.getId() == null) {
			// Don't write permissions for non-persisted objects
			return null;
		}
		
		Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		if (authentication == null) {
			return NO_PERMISSIONS;
		}
		
		Permissions perms = new Permissions();
		try {
			perms.isPublic = SecurityContextUtil.anyoneHasPermission(aclAwareModel, "READ");
		} catch (Throwable e) {
			LOG.warn("Could not read public permissions {}", e.getMessage(), e);
			perms.isPublic = false;
		}

		if (SecurityContextUtil.hasRole("ADMINISTRATOR")) {
			perms.grantAll();
		} else {
			try {
				perms.create = SecurityContextUtil.hasPermission(aclAwareModel, BasePermission.CREATE);
				perms.read = SecurityContextUtil.hasPermission(aclAwareModel, BasePermission.READ);
				perms.write = SecurityContextUtil.hasPermission(aclAwareModel, BasePermission.WRITE);
				perms.delete = SecurityContextUtil.hasPermission(aclAwareModel, BasePermission.DELETE);
				perms.manage = SecurityContextUtil.hasPermission(aclAwareModel, BasePermission.ADMINISTRATION);
			} catch (Throwable e) {
				LOG.warn("Could not read current permissions {}", e.getMessage(), e);
			}
		}
		return perms;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * com.fasterxml.jackson.databind.ser.VirtualBeanPropertyWriter#withConfig(com.
	 * fasterxml.jackson.databind.cfg.MapperConfig,
	 * com.fasterxml.jackson.databind.introspect.AnnotatedClass,
	 * com.fasterxml.jackson.databind.introspect.BeanPropertyDefinition,
	 * com.fasterxml.jackson.databind.JavaType)
	 */
	@Override
	public VirtualBeanPropertyWriter withConfig(MapperConfig<?> config, AnnotatedClass declaringClass, BeanPropertyDefinition propDef, JavaType type) {
		return new CurrentPermissionsWriter(propDef, declaringClass.getAnnotations(), type);
	}

}
