/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service.impl;

import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.NotUniqueUserException;
import org.genesys.blocks.security.UserException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.model.BasicUser.AccountType;
import org.genesys.blocks.security.persistence.AclEntryPersistence;
import org.genesys.blocks.security.service.BasicUserService;
import org.genesys.blocks.security.service.PasswordPolicy;
import org.genesys.blocks.security.service.PasswordPolicy.PasswordPolicyException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class BasicUserServiceImpl.
 *
 * @param <R> the generic type
 * @param <T> the generic type
 */
@Transactional(readOnly = true)
public abstract class BasicUserServiceImpl<R extends GrantedAuthority, T extends BasicUser<R>> implements BasicUserService<R, T>, InitializingBean {

	/** The Constant LOG. */
	public static final Logger LOG = LoggerFactory.getLogger(BasicUserServiceImpl.class);

	/** The Constant THIS_IS_NOT_A_PASSWORD. */
	/// A non-password used for system and Google accounts
	private static final String THIS_IS_NOT_A_PASSWORD = "THIS-IS-NOT-A-PASSWORD";

	/** The account lockout time. */
	private long accountLockoutTime = 5 * 60 * 1000;

	/** The user repository. */
	private JpaRepository<T, Long> _repository;

	/** The password encoder. */
	@Autowired
	protected PasswordEncoder passwordEncoder;

	/** The password policy. */
	@Autowired(required = false)
	private PasswordPolicy passwordPolicy;

	/** The acl entry repository. */
	@Autowired(required = false)
	protected AclEntryPersistence aclEntryRepository;

	@Override
	@Transactional
	public void afterPropertiesSet() throws Exception {
		this._repository = getUserRepository();
		try {
			loadUserByUsername(BasicUserService.SYSTEM_ADMIN);
		} catch (UsernameNotFoundException e) {
			T systemAdmin = createSystemAdministrator(BasicUserService.SYSTEM_ADMIN);

			if (systemAdmin == null) {
				throw new UserException("Implementation did not return a valid SYSTEM_ADMIN account");
			}
			if (systemAdmin.getAccountType() != AccountType.SYSTEM) {
				throw new UserException("Implementation did not return a SYSTEM_ADMIN account of type SYSTEM");
			}

			LOG.warn("New system admin {} account created with uuid={}", BasicUserService.SYSTEM_ADMIN, systemAdmin.getUuid());
		}
	}

	/**
	 * Gets the user repository.
	 *
	 * @return the user repository
	 */
	protected abstract JpaRepository<T, Long> getUserRepository();

	/**
	 * Implementations must create a user with specified username with ADMINISTRATOR
	 * role and account type {@link AccountType#SYSTEM}.
	 *
	 * @param username Generally SYSTEM_ADMIN
	 * @return user instance with ADMINISTRATOR role
	 * @throws UserException the user exception
	 */
	protected abstract T createSystemAdministrator(String username) throws UserException;

	/**
	 * Sets the account lockout time.
	 *
	 * @param accountLockoutTime the new account lockout time
	 */
	public void setAccountLockoutTime(final long accountLockoutTime) {
		this.accountLockoutTime = accountLockoutTime;
	}

	/* (non-Javadoc)
	 * @see org.genesys.blocks.security.service.BasicUserService#getDefaultUserRoles()
	 */
	@Override
	public abstract Collection<R> getDefaultUserRoles();
	
	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#listAvailableRoles()
	 */
	@Override
	public abstract List<R> listAvailableRoles();

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.core.userdetails.UserDetailsService#
	 * loadUserByUsername(java.lang.String)
	 */
	@Override
	public UserDetails loadUserByUsername(final String username) throws UsernameNotFoundException {
		T user = getUserByEmail(username);
		if (user == null) {
			throw new UsernameNotFoundException(username);
		}
		user.setRuntimeAuthorities(getRuntimeAuthorities(user));
		return user;
	}

	/**
	 * Allow the application to register additional authorities.
	 *
	 * @param user the user
	 * @return the same object
	 */
	protected abstract Set<String> getRuntimeAuthorities(T user);

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.BasicUserService#getUser(long)
	 */
	@Override
	public T getUser(final long id) {
		final T user = _repository.findOne(id);
		return deepLoad(user);
	}

	/**
	 * Deep load.
	 *
	 * @param user the user
	 * @return the t
	 */
	public T deepLoad(final T user) {
		if (user != null) {
			user.getRoles().size();
			user.getRoles().addAll(getDefaultUserRoles());
		}
		return user;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#updateUser(org.genesys.
	 * blocks.security.model.BasicUser, java.lang.String, java.lang.String)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') || principal.id == #user.id")
	public T updateUser(T user, final String email, final String fullName) throws NotUniqueUserException, UserException {
		// reload
		user = _repository.findOne(user.getId());

		if (!StringUtils.equals(email, user.getEmail()) && getUserByEmail(email) != null) {
			throw new NotUniqueUserException("Email address already registered");
		}

		user.setEmail(email);
		user.setFullName(fullName);

		return deepLoad(_repository.save(user));
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#deleteUser(org.genesys.
	 * blocks.security.model.BasicUser)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public void deleteUser(final T user) {
		_repository.delete(user);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#setRoles(org.genesys.
	 * blocks.security.model.BasicUser, java.util.Set)
	 */
	@Override
	@Transactional
	public T setRoles(T user, final Set<R> newRoles) {
		user = _repository.findOne(user.getId());

		// Remove transient roles
		newRoles.removeAll(getDefaultUserRoles());
		
		// If roles match, do nothing
		if (newRoles.containsAll(user.getRoles()) && user.getRoles().containsAll(newRoles)) {
			LOG.debug("Roles {} match {}. No change.", newRoles, user.getRoles());
			return user;
		}

		user.getRoles().clear();
		user.getRoles().addAll(newRoles);
		LOG.info("Setting roles for user {} to {}", user.getEmail(), user.getRoles());
		return deepLoad(_repository.save(user));
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.BasicUserService#changePassword(org.
	 * genesys.blocks.security.model.BasicUser, java.lang.String)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') || principal.id == #user.id")
	public T changePassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			setPassword(user, password);

			// Set account to expire 1 month after change password
			Calendar accountExpires = Calendar.getInstance();
			accountExpires.add(Calendar.MONTH, 1);
			user.setAccountExpires(accountExpires.getTime());

			return deepLoad(_repository.save(user));
		} else {
			throw new PasswordPolicyException("Password can be set only for LOCAL account types");
		}
	}

	/**
	 * Sets the password.
	 *
	 * @param user the user
	 * @param password the password
	 * @throws PasswordPolicyException the password policy exception
	 */
	protected final void setPassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			assureGoodPassword(password);
			user.setPassword(password == null ? null : passwordEncoder.encode(password));
			user.setPasswordExpires(null);
		} else {
			user.setPassword(THIS_IS_NOT_A_PASSWORD);
			user.setPasswordExpires(null);
		}
	}

	/**
	 * Test if password passes the password policy (if set).
	 *
	 * @param password candidate password
	 * @throws PasswordPolicyException if password does not match policy
	 */
	public void assureGoodPassword(final String password) throws PasswordPolicyException {
		if (passwordPolicy != null) {
			passwordPolicy.assureGoodPassword(password);
		}
	}

	/**
	 * For internal use only.
	 *
	 * @param userId the user id
	 * @param locked the locked
	 * @throws NoUserFoundException the no user found exception
	 */
	@Override
	@Transactional
	public void setAccountLockLocal(final long userId, final boolean locked) throws NoUserFoundException {
		final T user = getUser(userId);
		if (locked) {
			// Lock for account until some time
			user.setLockedUntil(new Date(System.currentTimeMillis() + accountLockoutTime));
			LOG.warn("Locking user account for user=" + user.getEmail() + "  until=" + user.getLockedUntil());
		} else {
			LOG.warn("Unlocking user account for user=" + user.getEmail());
			user.setLockedUntil(null);
		}
		_repository.save(user);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#setAccountLock(long,
	 * boolean)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public void setAccountLock(final long userId, final boolean locked) throws NoUserFoundException {
		setAccountLockLocal(userId, locked);
	}

	@Override
	@Transactional
	public T setAccountType(T user, AccountType accountType) {
		T u = _repository.findOne(user.getId());
		u.setAccountType(accountType);

		if (accountType != AccountType.LOCAL) {
			user.setPassword(THIS_IS_NOT_A_PASSWORD);
		}

		return _repository.save(u);
	}
	
	@Override
	@Transactional
	public void updateLastLogin(String userName) throws NoUserFoundException {
		T u = getUserByEmail(userName);
		if (u == null) {
			throw new NoUserFoundException("No such user.");
		}

		u.setLastLogin(new Date());

		// Set account to expire 1 year after last login
		Calendar accountExpires = Calendar.getInstance();
		accountExpires.add(Calendar.YEAR, 1);
		u.setAccountExpires(accountExpires.getTime());
		
		_repository.save(u);
	}
}
