/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service;

import java.util.ArrayList;
import java.util.List;

import org.genesys.blocks.util.JsonSidConverter;
import org.genesys.blocks.security.model.AclAwareModel;
import org.genesys.blocks.security.model.AclClass;
import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclObjectIdentity;
import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.security.serialization.AclEntriesToPermissions;
import org.genesys.blocks.security.serialization.Permissions;
import org.genesys.blocks.security.serialization.SidPermissions;
import org.springframework.security.acls.model.Permission;

import com.fasterxml.jackson.annotation.JsonUnwrapped;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

/**
 * The Interface CustomAclService.
 */
public interface CustomAclService extends JsonSidConverter.SidProvider {

	/**
	 * Gets the available permissions.
	 *
	 * @param className the class name
	 * @return the available permissions
	 */
	Permission[] getAvailablePermissions(String className);

	/**
	 * Get SID by ID.
	 *
	 * @param id -- ID of an AclSid entity
	 * @return persisted AclSid entity with specified id
	 */
	AclSid getSid(Long id);

	/**
	 * Gets the sid for the specified authority.
	 *
	 * @param authority the authority (must start with "ROLE_")
	 * @return the authority sid
	 */
	AclSid getAuthoritySid(String authority);

	/**
	 * Gets (and creates if missing) the sid for the specified authority.
	 *
	 * @param authority the authority (must start with "ROLE_")
	 * @return the authority sid
	 */
	AclSid ensureAuthoritySid(String authority);

	/**
	 * Removes the sid of the specified authority.
	 *
	 * @param authorityName the authority name
	 * @return the acl sid
	 */
	AclSid removeAuthoritySid(String authorityName);

	/**
	 * List authority sids.
	 *
	 * @return the list of {@link AclSid} for authorities
	 */
	List<AclSid> listAuthoritySids();

	/**
	 * Adds the creator permissions or updates permission inheritance.
	 *
	 * @param entity the target
	 */
	void createOrUpdatePermissions(AclAwareModel entity);

	/**
	 * Updates inheriting status of object identity.
	 *
	 * @param objectIdIdentity the id of object identity
	 * @param entriesInheriting the inheriting status
	 * @return the acl object identity
	 */
	AclObjectIdentity updateInheriting(long objectIdIdentity, boolean entriesInheriting);

	/**
	 * Updates parent object of object identity.
	 *
	 * @param objectIdIdentity the id of object identity
	 * @param parentObjectId the id of parent object identity
	 * @return the acl object identity
	 */
	AclObjectIdentity updateParentObject(long objectIdIdentity, long parentObjectId);

	/**
	 * Set ACL parent object for inherited permissions.
	 *
	 * @param target the target ACL object on which to change ACL
	 * @param parent the parent ACL object
	 * @return the acl object identity
	 */
	AclObjectIdentity setAclParent(AclAwareModel target, AclAwareModel parent);

	/**
	 * Removes the permissions on ACL model.
	 *
	 * @param target the target
	 */
	void removeAclAwareModel(AclAwareModel target);

	/**
	 * Removes the all permissions of SID.
	 *
	 * @param sid the sid
	 */
	void removePermissionsFor(AclSid sid);

	/**
	 * Get object identity by internal id.
	 *
	 * @param id AclObjectIdentity id
	 * @return the object identity
	 */
	AclObjectIdentity getObjectIdentity(long id);

	/**
	 * Gets the object identity for object of type className with specified id.
	 *
	 * @param id the id
	 * @param className the clazz
	 * @return the object identity
	 */
	AclObjectIdentity getObjectIdentity(long id, String className);

	/**
	 * Gets the object identity of the entity.
	 *
	 * @param entity the entity
	 * @return the object identity
	 */
	AclObjectIdentity getObjectIdentity(AclAwareModel entity);

	/**
	 * Gets the permissions.
	 *
	 * @param id the id
	 * @param className the class name
	 * @return the permissions
	 */
	List<SidPermissions> getPermissions(long id, String className);

	/**
	 * Gets the permissions.
	 *
	 * @param entity the entity
	 * @return the permissions
	 */
	List<SidPermissions> getPermissions(AclAwareModel entity);

	/**
	 * Gets the acl entries.
	 *
	 * @param objectIdentity the object identity
	 * @return the acl entries
	 */
	List<AclEntry> getAclEntries(AclObjectIdentity objectIdentity);

	/**
	 * Update permissions.
	 *
	 * @param entity the entity
	 * @param sid the sid
	 * @param permissions the permissions
	 * @return the acl object identity
	 */
	AclObjectIdentity setPermissions(AclAwareModel entity, AclSid sid, final Permissions permissions);

	/**
	 * Update permissions.
	 *
	 * @param objectIdentity the object identity
	 * @param sid the sid
	 * @param permissions the permissions
	 * @return the acl object identity
	 */
	AclObjectIdentity setPermissions(AclObjectIdentity objectIdentity, AclSid sid, final Permissions permissions);

	/**
	 * Removes the permissions for SID on ACL OID.
	 *
	 * @param objectIdentity the object identity
	 * @param aclSid the acl sid
	 * @return the acl object identity
	 */
	AclObjectIdentity removePermissions(AclObjectIdentity objectIdentity, AclSid aclSid);

	/**
	 * Gets the acl entries.
	 *
	 * @param entity the entity
	 * @return the acl entries
	 */
	List<AclEntry> getAclEntries(AclAwareModel entity);

	/**
	 * Gets the sids.
	 *
	 * @param id the id
	 * @param className the class name
	 * @return the sids
	 */
	List<AclSid> getSids(long id, String className);

	/**
	 * Gets the sids.
	 *
	 * @param entity the entity
	 * @return the sids
	 */
	List<AclSid> getSids(AclAwareModel entity);

	/**
	 * Ensure object identity.
	 * 
	 * @param id the object id identity
	 * @param className the class name
	 *
	 * @return the acl object identity
	 */
	AclObjectIdentity ensureObjectIdentity(long id, String className);

	/**
	 * List IDs of the specified class for the SID with specified permissions.
	 *
	 * @param clazz the clazz
	 * @param sid the sid
	 * @param permission the permission
	 * @return the list
	 */
	List<Long> listObjectIdentityIdsForSid(Class<? extends AclAwareModel> clazz, AclSid sid, Permission permission);

	/**
	 * Make entity publicly readable (or not).
	 *
	 * @param aclAwareModel The entity
	 * @param publiclyReadable true or false?
	 * @since 1.4
	 */
	void makePubliclyReadable(AclAwareModel aclAwareModel, boolean publiclyReadable);

	/**
	 * Cleanup ACL: remove {@link AclEntry} and {@link AclObjectIdentity} for
	 * missing ACL-aware entities, remove obsolete {@link AclClass}.
	 */
	void cleanupAcl();

	/**
	 * Gets the sid id.
	 *
	 * @param sid the sid
	 * @return the sid id
	 */
	Long getSidId(String sid);

	/**
	 * Gets the sid name.
	 *
	 * @param id the id
	 * @return the sid name
	 */
	String getSidName(long id);

	/**
	 * Load object identity extended information.
	 *
	 * @param objectIdentity the object identity
	 * @return the acl object identity ext
	 */
	AclObjectIdentityExt loadObjectIdentityExt(AclObjectIdentity objectIdentity);

	/**
	 * Wraps {@link AclObjectIdentity} and adds list of inherited permissions.
	 */
	public static class AclObjectIdentityExt {
		
		/** The original. */
		@JsonUnwrapped
		public AclObjectIdentity original;
		
		/** The inherited. */
		@JsonSerialize(converter = AclEntriesToPermissions.class)
		public List<AclEntry> inherited = new ArrayList<>();

		/**
		 * Instantiates a new acl object identity ext.
		 *
		 * @param source the source
		 */
		public AclObjectIdentityExt(AclObjectIdentity source) {
			this.original = source;
		}
	}
}
