/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.model;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.ElementCollection;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.MappedSuperclass;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Transient;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonView;

import org.genesys.blocks.model.JsonViews;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

/**
 * The Class BasicUser.
 *
 * When you extend this class, make sure you use:
 * 
 * <pre>@DiscriminatorValue(value = "1")</pre>
 *
 * @param <R> the generic type
 */
@MappedSuperclass
@DiscriminatorValue(value = "1")
public abstract class BasicUser<R extends GrantedAuthority> extends AclSid implements UserDetails {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -5318892732608111516L;

	/**
	 * The Enum AccountType.
	 */
	public static enum AccountType {

		/** The local. */
		LOCAL,
		/** The ldap. */
		LDAP,
		/** The google. */
		GOOGLE,
		/** The system. */
		SYSTEM,
		
		/** Deleted user accounts. */
		DELETED
	}

	/** The uuid. */
	@Column(length = 36, unique = true)
	private String uuid;

	/** The email. */
	@JsonView(JsonViews.Public.class)
	@Column(nullable = false, unique = true, length = 60)
	private String email;

	/** The short name. */
	@JsonView(JsonViews.Public.class)
	@Column(unique = true, length = 20)
	private String shortName;

	/** The full name. */
	@JsonView(JsonViews.Public.class)
	@Column(length = 60)
	private String fullName;

	/** The password. */
	@JsonIgnore
	@Column(length = 60)
	private String password;

	/** Account control. */
	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date accountExpires;

	/** The locked until. */
	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date lockedUntil;

	/** The password expires. */
	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date passwordExpires;

	/** The roles. */
	@JsonView(JsonViews.Protected.class)
	@ElementCollection(fetch = FetchType.EAGER)
	@Enumerated(EnumType.STRING)
	@CollectionTable(name = "userrole", joinColumns = @JoinColumn(name = "userId"))
	@Column(name = "role")
	private Set<R> roles = new HashSet<>();

	/** The account type. */
	@JsonView(JsonViews.Protected.class)
	@Enumerated(EnumType.STRING)
	@Column(length = 20)
	private AccountType accountType = AccountType.LOCAL;

	/** The date of last successful login. */
	@JsonView(JsonViews.Internal.class)
	@Temporal(TemporalType.TIMESTAMP)
	private Date lastLogin;

	@Transient
	@JsonIgnore
	private Set<String> runtimeAuthorities;

	/**
	 * Instantiates a new basic user.
	 */
	public BasicUser() {
		setPrincipal(true);
	}
	
	/**
	 * Ensure UUID.
	 */
	@PrePersist
	@PreUpdate
	void ensureUUID() {
		if (this.uuid == null) {
			this.uuid = UUID.randomUUID().toString();
		}

		// Use #email as SID name
		setSid(email);
	}

	/**
	 * Gets the email.
	 *
	 * @return the email
	 */
	public String getEmail() {
		return email;
	}

	/**
	 * Sets the email.
	 *
	 * @param email the new email
	 */
	public void setEmail(final String email) {
		this.email = email;
	}

	/**
	 * Gets the short name.
	 *
	 * @return the short name
	 */
	public String getShortName() {
		return shortName;
	}

	/**
	 * Sets the short name.
	 *
	 * @param shortName the new short name
	 */
	public void setShortName(final String shortName) {
		this.shortName = shortName;
	}

	/**
	 * Gets the full name.
	 *
	 * @return the full name
	 */
	@Override
	public String getFullName() {
		return fullName;
	}

	/**
	 * Sets the full name.
	 *
	 * @param fullName the new full name
	 */
	public void setFullName(final String fullName) {
		this.fullName = fullName;
	}

	/**
	 * Gets the account expires.
	 *
	 * @return the account expires
	 */
	public Date getAccountExpires() {
		return accountExpires;
	}

	/**
	 * Sets the account expires.
	 *
	 * @param accountExpires the new account expires
	 */
	public void setAccountExpires(final Date accountExpires) {
		this.accountExpires = accountExpires;
	}

	/**
	 * Gets the locked until.
	 *
	 * @return the locked until
	 */
	public Date getLockedUntil() {
		return lockedUntil;
	}

	/**
	 * Sets the locked until.
	 *
	 * @param lockedUntil the new locked until
	 */
	public void setLockedUntil(final Date lockedUntil) {
		this.lockedUntil = lockedUntil;
	}

	/**
	 * Gets the password expires.
	 *
	 * @return the password expires
	 */
	public Date getPasswordExpires() {
		return passwordExpires;
	}

	/**
	 * Sets the password expires.
	 *
	 * @param passwordExpires the new password expires
	 */
	public void setPasswordExpires(final Date passwordExpires) {
		this.passwordExpires = passwordExpires;
	}

	/**
	 * Sets the password.
	 *
	 * @param password the new password
	 */
	public void setPassword(final String password) {
		this.password = password;
	}

	/**
	 * Gets the roles.
	 *
	 * @return the roles
	 */
	public Set<R> getRoles() {
		return roles;
	}

	/**
	 * Sets the roles.
	 *
	 * @param roles the new roles
	 */
	public void setRoles(final Set<R> roles) {
		this.roles = roles;
	}

	/**
	 * Gets the uuid.
	 *
	 * @return the uuid
	 */
	public String getUuid() {
		return uuid;
	}

	/**
	 * Sets the uuid.
	 *
	 * @param uuid the new uuid
	 */
	public void setUuid(final String uuid) {
		this.uuid = uuid;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return MessageFormat.format("id={0} email={1} fullName={2}", getId(), email, fullName);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.core.userdetails.UserDetails#getAuthorities()
	 */
	@Transient
	@Override
	public Collection<? extends GrantedAuthority> getAuthorities() {
		Set<SimpleGrantedAuthority> authorities = new HashSet<>();
		authorities.addAll(getRoles().stream().map(role -> new SimpleGrantedAuthority(role.getAuthority())).collect(Collectors.toSet()));
		if (runtimeAuthorities != null) {
			authorities.addAll(runtimeAuthorities.stream().map(SimpleGrantedAuthority::new).collect(Collectors.toSet()));
		}
		return authorities;
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.core.userdetails.UserDetails#getPassword()
	 */
	@Override
	public String getPassword() {
		return password;
	}

	/**
	 * {@link #getUsername()} must return the name used by UserDetailService in the {@link org.springframework.security.core.userdetails.UserDetailsService#loadUserByUsername(String)}
	 * 
	 * @see org.springframework.security.core.userdetails.UserDetails#getUsername()
	 */
	@Override
	public String getUsername() {
		return email;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.core.userdetails.UserDetails#isAccountNonExpired
	 * ()
	 */
	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isAccountNonExpired() {
		return true;
		// TODO Re-enable when we have a way to extend accounts
//		return (accountExpires == null) || !accountExpires.before(new Date());
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.core.userdetails.UserDetails#isAccountNonLocked(
	 * )
	 */
	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isAccountNonLocked() {
		return (lockedUntil == null) || !lockedUntil.after(new Date());
	}

	/**
	 * Checks if is account locked.
	 *
	 * @return true, if is account locked
	 */
	@JsonView(JsonViews.Protected.class)
	public boolean isAccountLocked() {
		return !isAccountNonLocked();
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.core.userdetails.UserDetails#
	 * isCredentialsNonExpired()
	 */
	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isCredentialsNonExpired() {
		return (passwordExpires == null) || !passwordExpires.before(new Date());
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.core.userdetails.UserDetails#isEnabled()
	 */
	@Override
	@JsonView(JsonViews.Protected.class)
	public boolean isEnabled() {
		return super.isActive();
	}

	/**
	 * Gets the account type.
	 *
	 * @return the account type
	 */
	public AccountType getAccountType() {
		return accountType;
	}

	/**
	 * Sets the account type.
	 *
	 * @param accountType the new account type
	 */
	public void setAccountType(final AccountType accountType) {
		this.accountType = accountType;
	}

	/**
	 * Gets the last login.
	 *
	 * @return the last login
	 */
	public Date getLastLogin() {
		return lastLogin;
	}

	/**
	 * Sets the last login.
	 *
	 * @param lastLogin the new last login
	 */
	public void setLastLogin(Date lastLogin) {
		this.lastLogin = lastLogin;
	}
	
	/**
	 * Set additional authorities at runtime.
	 *
	 * @param authorities the new runtime authorities
	 */
	public void setRuntimeAuthorities(Set<String> authorities) {
		this.runtimeAuthorities = authorities;
	}
}
