/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.service;

import java.util.List;

import org.genesys.blocks.oauth.model.OAuthClient;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.oauth2.provider.ClientDetailsService;

/**
 * The Interface OAuthClientDetailsService.
 */
public interface OAuthClientDetailsService extends ClientDetailsService {

	/**
	 * List client details.
	 *
	 * @return the list
	 */
	List<OAuthClient> listClientDetails();

	/**
	 * List client details.
	 *
	 * @param pageable the pageable
	 * @return page with Client details
	 */
	Page<OAuthClient> listClientDetails(Pageable pageable);

	/**
	 * Gets the client.
	 *
	 * @param clientId the client id
	 * @return the client
	 */
	OAuthClient getClient(String clientId);

	/**
	 * Register a new OAuth client.
	 *
	 * @param client the client
	 * @return the OAuth client
	 * @since 1.3
	 */
	OAuthClient addClient(OAuthClient client);

	/**
	 * Update client.
	 *
	 * @param id the id
	 * @param version the version
	 * @param updates the updates
	 * @return the o auth client
	 */
	OAuthClient updateClient(long id, int version, OAuthClient updates);

	/**
	 * Removes the client.
	 *
	 * @param oauthClient the oauth client
	 * @return the o auth client
	 */
	OAuthClient removeClient(OAuthClient oauthClient);

	/**
	 * Autocomplete OAuth clients by title.
	 *
	 * @param title the title
	 * @param limit maximum number of results
	 * @return list of auth clients
	 */
	List<OAuthClient> autocompleteClients(String title, int limit);

	/**
	 * Generates a new clientSecret.
	 *
	 * @param oauthClient the client to regenerate secret for
	 * @return the new cliet secret
	 */
	String resetSecret(OAuthClient oauthClient);

	/**
	 * Removes the Client secret.
	 *
	 * @param oauthClient the oauth client
	 * @return Updated OAuth client
	 */
	OAuthClient removeSecret(OAuthClient oauthClient);

}
