/*
 * Copyright 2018 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.oauth.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.Lob;
import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Transient;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.StringUtils;
import org.genesys.blocks.model.Copyable;
import org.genesys.blocks.model.JsonViews;
import org.genesys.blocks.security.model.AclSid;
import org.hibernate.annotations.Type;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.provider.ClientDetails;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonView;

/**
 * OAuth Client information.
 *
 * @author Matija Obreza
 */
@Entity
@Table(name = "oauthclient")
@DiscriminatorValue(value = "2")
public class OAuthClient extends AclSid implements ClientDetails, Copyable<OAuthClient> {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -4204753722663196007L;

	/** The client id. */
	@JsonView(JsonViews.Public.class)
	@Column(unique = true, nullable = false, updatable = false, length = 100)
	private String clientId;

	/** The client secret. */
	@JsonView(JsonViews.Protected.class)
	@Column(nullable = true, length = 100)
	private String clientSecret;

	/** The resource. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String resource;

	/** The resource ids. */
	@JsonView(JsonViews.Protected.class)
	@Transient
	private Set<String> resourceIds = new HashSet<>();

	/** The auto approve. */
	@Column(nullable = false)
	boolean autoApprove = false;

	/** The auto approve scope. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String autoApproveScope;

	/** The auto approve scopes. */
	@JsonView(JsonViews.Protected.class)
	@Transient
	private Set<String> autoApproveScopes = new HashSet<>();

	/** The scope. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String scope;

	/** The scopes. */
	@JsonView(JsonViews.Protected.class)
	@Transient
	private Set<String> scopes = new HashSet<>();

	/** The grants. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String grants;

	/** The grant types. */
	@JsonView(JsonViews.Protected.class)
	@Transient
	private Set<String> grantTypes = new HashSet<>();

	/** The redirect. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String redirect;

	/** The redirect uris. */
	@JsonView(JsonViews.Protected.class)
	@Transient
	private Set<String> redirectUris = new HashSet<>();

	/** The roles. */
	@JsonView(JsonViews.Protected.class)
	@ElementCollection(fetch = FetchType.EAGER)
	@Enumerated(EnumType.STRING)
	@CollectionTable(name = "oauthclientrole", joinColumns = @JoinColumn(name = "clientId"))
	@Column(name = "oauthclientrole")
	private Set<OAuthRole> roles = new HashSet<>();

	/** The additional information. */
	@Transient
	private Map<String, Object> additionalInformation = null;

	/** The access token validity. */
	private Integer accessTokenValidity;

	/** The refresh token validity. */
	private Integer refreshTokenValidity;

	/** The title. */
	@Column(nullable = false, length = 200)
	private String title;

	/** The description. */
	@Lob
	@Type(type = "org.hibernate.type.TextType")
	private String description;

	/** Allowed CORS origins */
	@Transient
	@JsonView(JsonViews.Protected.class)
	private Set<String> allowedOrigins = new HashSet<>();

	/** The origins. */
	@JsonIgnore
	@Column(nullable = true, length = 200)
	private String origins;

	/**
	 * Instantiates a new o auth client.
	 */
	public OAuthClient() {
		setPrincipal(true);
	}

	@PrePersist
	private void assignSid() {
		flatten();

		// Use clientId as SID name
		setSid(clientId);
	}

	/**
	 * Flatten.
	 */
	@PreUpdate
	private void flatten() {
		resource = resourceIds.stream().collect(Collectors.joining(";"));
		scope = scopes.stream().collect(Collectors.joining(";"));
		autoApproveScope = autoApproveScopes.stream().collect(Collectors.joining(";"));
		grants = grantTypes.stream().collect(Collectors.joining(";"));
		redirect = redirectUris.stream().collect(Collectors.joining(";"));
		origins = allowedOrigins.stream().collect(Collectors.joining(";"));
	}

	/**
	 * Inflate.
	 */
	@PostLoad
	private void inflate() {
		if (resource != null) {
			Arrays.stream(StringUtils.split(resource, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(r -> resourceIds.add(r));
		}
		if (scope != null) {
			Arrays.stream(StringUtils.split(scope, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(s -> scopes.add(s));
		}
		if (autoApproveScope != null) {
			Arrays.stream(StringUtils.split(autoApproveScope, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(s -> autoApproveScopes.add(s));
		}
		if (grants != null) {
			Arrays.stream(StringUtils.split(grants, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(g -> grantTypes.add(g));
		}
		if (redirect != null) {
			Arrays.stream(StringUtils.split(redirect, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(u -> redirectUris.add(u));
		}
		if (origins != null) {
			Arrays.stream(StringUtils.split(origins, ";")).filter(r -> StringUtils.isNotBlank(r)).forEach(u -> allowedOrigins.add(u));
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#getClientId()
	 */
	@Override
	public String getClientId() {
		return clientId;
	}

	/**
	 * Sets the client id.
	 *
	 * @param clientId the new client id
	 */
	public void setClientId(final String clientId) {
		this.clientId = clientId;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.oauth2.provider.ClientDetails#getClientSecret()
	 */
	@Override
	public String getClientSecret() {
		return clientSecret;
	}

	/**
	 * Sets the client secret.
	 *
	 * @param clientSecret the new client secret
	 */
	public void setClientSecret(final String clientSecret) {
		this.clientSecret = clientSecret;
	}

	/**
	 * Gets the resource.
	 *
	 * @return the resource
	 */
	public String getResource() {
		return resource;
	}

	/**
	 * Sets the resource.
	 *
	 * @param resource the new resource
	 */
	public void setResource(final String resource) {
		this.resource = resource;
	}

	/**
	 * Gets the grants.
	 *
	 * @return the grants
	 */
	public String getGrants() {
		return grants;
	}

	/**
	 * Sets the grants.
	 *
	 * @param grants the new grants
	 */
	public void setGrants(final String grants) {
		this.grants = grants;
	}

	/**
	 * Gets the redirect.
	 *
	 * @return the redirect
	 */
	public String getRedirect() {
		return redirect;
	}

	/**
	 * Sets the redirect.
	 *
	 * @param redirect the new redirect
	 */
	public void setRedirect(final String redirect) {
		this.redirect = redirect;
	}

	/**
	 * Gets the roles.
	 *
	 * @return the roles
	 */
	public Collection<OAuthRole> getRoles() {
		return roles;
	}

	/**
	 * Sets the roles.
	 *
	 * @param roles the new roles
	 */
	public void setRoles(final Set<OAuthRole> roles) {
		this.roles = roles;
	}

	/**
	 * Sets the scope.
	 *
	 * @param scope the new scope
	 */
	public void setScope(final String scope) {
		this.scope = scope;
	}

	/**
	 * Sets the auto approve.
	 *
	 * @param autoApprove the new auto approve
	 */
	public void setAutoApprove(final boolean autoApprove) {
		this.autoApprove = autoApprove;
	}

	/**
	 * Gets the auto approve.
	 *
	 * @return the auto approve
	 */
	public boolean getAutoApprove() {
		return autoApprove;
	}

	/**
	 * Gets the auto approve scope.
	 *
	 * @return the auto approve scope
	 */
	protected String getAutoApproveScope() {
		return autoApproveScope;
	}

	/**
	 * Sets the auto approve scope.
	 *
	 * @param autoApproveScope the new auto approve scope
	 */
	protected void setAutoApproveScope(final String autoApproveScope) {
		this.autoApproveScope = autoApproveScope;
	}

	/**
	 * Gets the auto approve scopes.
	 *
	 * @return the auto approve scopes
	 */
	public Set<String> getAutoApproveScopes() {
		return autoApproveScopes;
	}
	
	/**
	 * Sets the auto approve scopes.
	 *
	 * @param autoApproveScopes the new auto approve scopes
	 */
	public void setAutoApproveScopes(Set<String> autoApproveScopes) {
		this.autoApproveScopes = autoApproveScopes;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.oauth2.provider.ClientDetails#getResourceIds()
	 */
	@Override
	public Set<String> getResourceIds() {
		return resourceIds;
	}
	
	/**
	 * Sets the resource ids.
	 *
	 * @param resourceIds the new resource ids
	 */
	public void setResourceIds(Set<String> resourceIds) {
		this.resourceIds = resourceIds;
	}

	/**
	 * Client secret is required when provided.
	 *
	 * @return true, if is secret required
	 */
	@Override
	public boolean isSecretRequired() {
		return clientSecret != null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#isScoped()
	 */
	@Override
	public boolean isScoped() {
		return !scopes.isEmpty();
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#getScope()
	 */
	@JsonProperty("clientScopes")
	@Override
	public Set<String> getScope() {
		return scopes;
	}
	
	/**
	 * Sets the scopes.
	 *
	 * @param scopes the new scopes
	 */
	@JsonProperty("clientScopes")
	public void setScopes(Set<String> scopes) {
		this.scopes = scopes;
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#
	 * getAuthorizedGrantTypes()
	 */
	@Override
	public Set<String> getAuthorizedGrantTypes() {
		return grantTypes;
	}
	
	/**
	 * Sets the grant types.
	 *
	 * @param grantTypes the new grant types
	 */
	public void setGrantTypes(Set<String> grantTypes) {
		this.grantTypes = grantTypes;
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#
	 * getRegisteredRedirectUri()
	 */
	@Override
	public Set<String> getRegisteredRedirectUri() {
		return redirectUris;
	}

	/**
	 * Sets the redirect uris.
	 *
	 * @param redirectUris the new redirect uris
	 */
	public void setRedirectUris(Set<String> redirectUris) {
		this.redirectUris = redirectUris;
	}
	
	
	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.oauth2.provider.ClientDetails#getAuthorities()
	 */
	@Override
	public Collection<GrantedAuthority> getAuthorities() {
		return roles.stream().collect(Collectors.toList());
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#
	 * getAccessTokenValiditySeconds()
	 */
	@Override
	public Integer getAccessTokenValiditySeconds() {
		return accessTokenValidity;
	}

	/**
	 * Gets the access token validity.
	 *
	 * @return the access token validity
	 */
	public Integer getAccessTokenValidity() {
		return accessTokenValidity;
	}

	/**
	 * Sets the access token validity.
	 *
	 * @param accessTokenValidity the new access token validity
	 */
	public void setAccessTokenValidity(final Integer accessTokenValidity) {
		this.accessTokenValidity = accessTokenValidity;
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#
	 * getRefreshTokenValiditySeconds()
	 */
	@Override
	public Integer getRefreshTokenValiditySeconds() {
		return refreshTokenValidity;
	}

	/**
	 * Gets the refresh token validity.
	 *
	 * @return the refresh token validity
	 */
	public Integer getRefreshTokenValidity() {
		return refreshTokenValidity;
	}

	/**
	 * Sets the refresh token validity.
	 *
	 * @param refreshTokenValidity the new refresh token validity
	 */
	public void setRefreshTokenValidity(final Integer refreshTokenValidity) {
		this.refreshTokenValidity = refreshTokenValidity;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.springframework.security.oauth2.provider.ClientDetails#isAutoApprove(java
	 * .lang.String)
	 */
	@Override
	public boolean isAutoApprove(final String scope) {
		return autoApprove || autoApproveScopes.contains(scope);
	}

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.oauth2.provider.ClientDetails#
	 * getAdditionalInformation()
	 */
	@Override
	public Map<String, Object> getAdditionalInformation() {
		return additionalInformation;
	}
	
	/**
	 * Sets the additional information.
	 *
	 * @param additionalInformation the additional information
	 */
	public void setAdditionalInformation(Map<String, Object> additionalInformation) {
		this.additionalInformation = additionalInformation;
	}
	
	/**
	 * Sets the title.
	 *
	 * @param title the new title
	 */
	public void setTitle(final String title) {
		this.title = title;
	}

	/**
	 * Gets the title.
	 *
	 * @return the title
	 */
	public String getTitle() {
		return title;
	}

	/**
	 * Sets the description.
	 *
	 * @param description the new description
	 */
	public void setDescription(final String description) {
		this.description = description;
	}

	/**
	 * Gets the description.
	 *
	 * @return the description
	 */
	public String getDescription() {
		return description;
	}

	@Override
	public String getFullName() {
		return this.clientId;
	}

	/**
	 * Returns null.
	 *
	 * @return the o auth client
	 */
	@Override
	public OAuthClient copy() {
		// Unsupported
		return null;
	}

	@Override
	public OAuthClient apply(OAuthClient source) {
		String oldSecret = this.clientSecret;
		Copyable.super.apply(source);
		// Keep old secret
		this.clientSecret = oldSecret;

		this.autoApproveScopes.clear();
		this.autoApproveScopes.addAll(source.autoApproveScopes);

		this.grantTypes.clear();
		this.grantTypes.addAll(source.grantTypes);

		this.redirectUris.clear();
		this.redirectUris.addAll(source.redirectUris);

		this.resourceIds.clear();
		this.resourceIds.addAll(source.resourceIds);

		this.roles.clear();
		this.roles.addAll(source.roles);

		this.scopes.clear();
		this.scopes.addAll(source.scopes);

		this.allowedOrigins.clear();
		this.allowedOrigins.addAll(source.allowedOrigins);

		return this;
	}

	public String getOrigins() {
		return origins;
	}
	
	public void setOrigins(String origins) {
		this.origins = origins;
	}
	
	public Set<String> getAllowedOrigins() {
		return allowedOrigins;
	}
	
	public void setAllowedOrigins(Set<String> allowedOrigins) {
		this.allowedOrigins = allowedOrigins;
	}
}
