/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.genesys.blocks.security.SecurityContextUtil;
import org.genesys.blocks.security.model.AclAwareModel;
import org.genesys.blocks.security.model.AclClass;
import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclObjectIdentity;
import org.genesys.blocks.security.model.AclSid;
import org.genesys.blocks.security.persistence.AclClassPersistence;
import org.genesys.blocks.security.persistence.AclEntryPersistence;
import org.genesys.blocks.security.persistence.AclObjectIdentityPersistence;
import org.genesys.blocks.security.persistence.AclSidPersistence;
import org.genesys.blocks.security.service.CustomAclService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.acls.domain.BasePermission;
import org.springframework.security.acls.model.Permission;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class CustomAclServiceImpl.
 */
@Service
@Transactional
public class CustomAclServiceImpl implements CustomAclService {

	/** The base permissions. */
	private static Permission[] basePermissions;

	/** The acl object identity persistence. */
	@Autowired
	private AclObjectIdentityPersistence aclObjectIdentityPersistence;

	/** The acl class persistence. */
	@Autowired
	private AclClassPersistence aclClassPersistence;

	/** The acl entry persistence. */
	@Autowired
	private AclEntryPersistence aclEntryPersistence;

	/** The cache manager. */
	@Autowired
	private CacheManager cacheManager;

	/** The acl sid persistence. */
	@Autowired
	private AclSidPersistence aclSidPersistence;

	static {
		basePermissions = new Permission[] { BasePermission.CREATE, BasePermission.READ, BasePermission.WRITE, BasePermission.DELETE, BasePermission.ADMINISTRATION };
	}

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(CustomAclServiceImpl.class);

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#blankPermissionsMap()
	 */
	@Override
	public Map<Integer, Boolean> blankPermissionsMap() {
		HashMap<Integer, Boolean> map = new HashMap<>();
		for (Permission p : basePermissions) {
			map.put(p.getMask(), false);
		}
		return map;
	}

	@Override
	@Transactional(readOnly = true)
	public AclSid getSid(Long id) {
		return aclSidPersistence.findOne(id);
	}

	@Override
	public AclSid getAuthoritySid(String authority) {
		return ensureSidForAuthority(authority);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#addCreatorPermissions(
	 * org.genesys.blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRED)
	public void addCreatorPermissions(final AclAwareModel target) {
		if ((target == null) || (target.getId() <= 0l)) {
			LOG.warn("No target specified for ACL permissions, bailing out!");
			return;
		}

		final AclSid ownerSid = SecurityContextUtil.getCurrentUser();
		if (ownerSid == null) {
			LOG.warn("No SID in security context, not assigning creator permissions");
			return;
		} else if (!ownerSid.isPersisted()) {
			LOG.warn("Owner SID not persisted, not assigning creator permissions");
			return;
		}

		LOG.debug("Inserting owner ACL entries for owner={} class={} id={}", ownerSid, target.getClass().getName(), target.getId());

		final AclClass aclClass = ensureAclClass(target.getClass().getName());

		// create object identity
		final AclObjectIdentity objectIdentity = new AclObjectIdentity();
		objectIdentity.setObjectIdIdentity(target.getId());
		objectIdentity.setAclClass(aclClass);
		objectIdentity.setOwnerSid(ownerSid);
		objectIdentity.setParentObject(null);
		objectIdentity.setEntriesInheriting(false);

		// save object identity
		AclObjectIdentity savedAclObjectIdentity = aclObjectIdentityPersistence.findByIdAndClassname(objectIdentity.getObjectIdIdentity(), objectIdentity.getAclClass().getAclClass());
		if (savedAclObjectIdentity == null) {
			savedAclObjectIdentity = aclObjectIdentityPersistence.save(objectIdentity);
			final Map<Integer, Boolean> permissionsMap = new HashMap<>();
			for (final Permission permission : basePermissions) {
				permissionsMap.put(permission.getMask(), true);
			}

			addPermissions(ownerSid, savedAclObjectIdentity, permissionsMap);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#removePermissions(org.
	 * genesys.blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRED)
	public void removePermissions(final AclAwareModel target) {
		LOG.debug("Deleting all ACL entries for {}", target);
		final AclObjectIdentity savedAclObjectIdentity = getObjectIdentity(target);
		if (savedAclObjectIdentity != null) {
			final List<AclEntry> aclEntries = aclEntryPersistence.findByObjectIdentity(savedAclObjectIdentity);
			if (aclEntries != null) {
				aclEntryPersistence.delete(aclEntries);
				aclObjectIdentityPersistence.delete(savedAclObjectIdentity.getId());
			}
			clearAclCache();
		}
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#removePermissions(org.
	 * genesys.blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(propagation = Propagation.REQUIRED)
	public void removePermissionsFor(final AclSid sid) {
		int count = aclEntryPersistence.deleteForSid(sid);
		LOG.debug("Deleting {} permision entries granted to {}", count, sid);
		if (count > 0) {
			clearAclCache();
		}
	}

	/**
	 * Adds the permissions.
	 *
	 * @param sid the sid
	 * @param objectIdentity the object identity
	 * @param permissions the permissions
	 * @return
	 */
	private boolean addPermissions(final AclSid sid, final AclObjectIdentity objectIdentity, final Map<Integer, Boolean> permissions) {
		// create Acl Entry
		for (final Permission permission : basePermissions) {
			final int mask = permission.getMask();
			final AclEntry aclEntry = new AclEntry();
			aclEntry.setAclObjectIdentity(objectIdentity);
			aclEntry.setAclSid(sid);
			aclEntry.setAceOrder(getAceOrder(objectIdentity.getId()));
			aclEntry.setGranting(permissions.get(mask));
			aclEntry.setAuditSuccess(true);
			aclEntry.setAuditFailure(true);
			// set full access for own organization
			aclEntry.setMask(mask);

			// save ACL
			aclEntryPersistence.save(aclEntry);
		}

		clearAclCache();
		return true;
	}

	private void clearAclCache() {
		final Cache aclCache = cacheManager.getCache("aclCache");
		if (aclCache != null)
			aclCache.clear();
	}

	/**
	 * Generates next ace_order value (to avoid DuplicateIndex exception :
	 * acl_object_identity + ace_order is unique index).
	 *
	 * @param aclObjectEntityId - id of acl_object_identity table
	 * @return - ace_order value
	 */
	private Long getAceOrder(final long aclObjectEntityId) {
		final Long maxAceOrder = aclEntryPersistence.getMaxAceOrderForObjectEntity(aclObjectEntityId);
		return maxAceOrder != null ? maxAceOrder + 1 : 1;
	}

	/**
	 * Ensure acl class.
	 *
	 * @param className the class name
	 * @return the acl class
	 */
	private AclClass ensureAclClass(final String className) {
		AclClass aclClass = aclClassPersistence.findByAclClass(className);

		if (aclClass == null) {
			LOG.warn("Missing AclClass '{}'", className);
			aclClass = new AclClass();
			aclClass.setAclClass(className);
			return aclClassPersistence.save(aclClass);
		}

		return aclClass;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#getObjectIdentity(java.
	 * lang.String, long)
	 */
	@Override
	@Transactional(readOnly = true)
	public AclObjectIdentity getObjectIdentity(final String className, final long id) {
		return aclObjectIdentityPersistence.findByIdAndClassname(id, className);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#getObjectIdentity(org.
	 * genesys.blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(readOnly = true)
	public AclObjectIdentity getObjectIdentity(final AclAwareModel entity) {
		if (entity == null) {
			LOG.error("getObjectIdentity: Entity is null");
		}
		final AclObjectIdentity oid = aclObjectIdentityPersistence.findByIdAndClassname(entity.getId(), entity.getClass().getName());
		if (oid == null) {
			LOG.warn("ACL object identity not found for class={} id={}", entity.getClass().getName(), entity.getId());
		}
		return oid;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#getAvailablePermissions(
	 * java.lang.String)
	 */
	@Override
	@Transactional(readOnly = true)
	public Permission[] getAvailablePermissions(final String className) {
		// Do not remove parameter. We may change available permissions based on
		// parameter type!
		return basePermissions;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#getPermissions(long,
	 * java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#id, #className, 'ADMINISTRATION')")
	public Map<String, Map<Integer, Boolean>> getPermissions(final long id, final String className) {
		final Map<String, Map<Integer, Boolean>> perm = new HashMap<>();

		final List<AclEntry> aclEntries = getAclEntries(getObjectIdentity(className, id));
		for (final AclEntry aclEntry : aclEntries) {
			Map<Integer, Boolean> granted = perm.get(aclEntry.getAclSid().getSid());
			if (granted == null) {
				perm.put(aclEntry.getAclSid().getSid(), granted = new HashMap<>());
			}
			granted.put((int) aclEntry.getMask(), aclEntry.isGranting());
		}

		return perm;
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.CustomAclService#getPermissions(org.
	 * genesys.blocks.security.model.AclAwareModel)
	 */
	@Transactional(readOnly = true)
	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity, 'ADMINISTRATION')")
	public Map<String, Map<Integer, Boolean>> getPermissions(final AclAwareModel entity) {
		return getPermissions(entity.getId(), entity.getClass().getName());
	}

	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity, 'ADMINISTRATION')")
	public void updatePermissions(final AclAwareModel entity, final AclSid sid, final Map<Integer, Boolean> permissions) {
		final AclObjectIdentity objectIdentity = getObjectIdentity(entity);
		updatePermissions(objectIdentity, sid, permissions);
	}

	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#objectIdentity.objectIdIdentity, #objectIdentity.aclClass.aclClass, 'ADMINISTRATION')")
	public void updatePermissions(AclObjectIdentity objectIdentity, AclSid sid, Map<Integer, Boolean> permissions) {
		boolean oneGranting = false;
		final List<AclEntry> aclEntries = aclEntryPersistence.findBySidAndObjectIdentity(sid, objectIdentity);
		for (final AclEntry aclEntry : aclEntries) {
			aclEntry.setGranting(permissions.get((int) aclEntry.getMask()));
			oneGranting |= aclEntry.isGranting();
		}
		if (oneGranting) {
			LOG.info("Saving " + aclEntries);
			aclEntryPersistence.save(aclEntries);
		} else {
			LOG.info("Deleting " + aclEntries);
			aclEntryPersistence.delete(aclEntries);
		}

		clearAclCache();
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.CustomAclService#getAclEntries(org.
	 * genesys.blocks.security.model.AclObjectIdentity)
	 */
	@Override
	@Transactional(readOnly = true)
    @PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#objectIdentity.objectIdIdentity, #objectIdentity.aclClass.aclClass, 'ADMINISTRATION')")
	public List<AclEntry> getAclEntries(final AclObjectIdentity objectIdentity) {
		return aclEntryPersistence.findByObjectIdentity(objectIdentity);
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.CustomAclService#getAclEntries(org.
	 * genesys.blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(readOnly = true)
	public List<AclEntry> getAclEntries(final AclAwareModel entity) {
		return aclEntryPersistence.findByObjectIdentity(getObjectIdentity(entity));
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.CustomAclService#getSids(long,
	 * java.lang.String)
	 */
	@Override
	@Transactional(readOnly = true)
	public List<AclSid> getSids(final long id, final String className) {
		return aclEntryPersistence.getSids(getObjectIdentity(className, id));
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#getSids(org.genesys.
	 * blocks.security.model.AclAwareModel)
	 */
	@Override
	@Transactional(readOnly = true)
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#entity, 'ADMINISTRATION')")
	public List<AclSid> getSids(final AclAwareModel entity) {
		return aclEntryPersistence.getSids(getObjectIdentity(entity));
	}

	@Override
	public boolean addPermissions(AclAwareModel entity, AclSid sid, Map<Integer, Boolean> permissions) {
		return addPermissions(sid, getObjectIdentity(entity), permissions);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#addPermissions(long,
	 * java.lang.String, java.lang.String, boolean, java.util.Map)
	 */
	@Override
	@PreAuthorize("hasRole('ADMINISTRATOR') or hasPermission(#objectIdIdentity, #className, 'ADMINISTRATION')")
	public boolean addPermissions(final long objectIdIdentity, final String className, AclSid sid, final Map<Integer, Boolean> permissions) {
		final AclObjectIdentity oid = ensureObjectIdentity(className, objectIdIdentity);

		return addPermissions(sid, oid, permissions);
	}

	/**
	 * Ensure ACL SID entry for the specified authority name (role)
	 */
	private AclSid ensureSidForAuthority(String authority) {
		AclSid roleSid = aclSidPersistence.findBySidAndPrincipal(authority, false);

		if (roleSid == null) {
			LOG.warn("Creating AclSid for role '{}'", authority);
			roleSid = new AclSid();
			roleSid.setPrincipal(false);
			roleSid.setSid(authority);
			return aclSidPersistence.save(roleSid);
		}

		return roleSid;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#ensureObjectIdentity(
	 * java.lang.String, long)
	 */
	@Override
	@Transactional
	public AclObjectIdentity ensureObjectIdentity(final String className, final long objectIdIdentity) {
		AclObjectIdentity aoi = aclObjectIdentityPersistence.findByIdAndClassname(objectIdIdentity, className);
		if (aoi == null) {
			aoi = new AclObjectIdentity();
			aoi.setObjectIdIdentity(objectIdIdentity);
			aoi.setAclClass(ensureAclClass(className));
			final AclSid ownerSid = SecurityContextUtil.getCurrentUser();
			aoi.setOwnerSid(ownerSid);
			aoi = aclObjectIdentityPersistence.save(aoi);
		}
		return aoi;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.CustomAclService#listIdentitiesForSid(
	 * java.lang.Class, org.springframework.security.core.userdetails.UserDetails,
	 * org.springframework.security.acls.model.Permission)
	 */
	@Override
	@Transactional(readOnly = true)
	public List<Long> listObjectIdentityIdsForSid(final Class<? extends AclAwareModel> clazz, final AclSid sid, final Permission permission) {
		return aclEntryPersistence.findObjectIdentitiesForSidAndAclClassAndMask(sid, clazz.getName(), permission.getMask());
	}

}
