/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service.impl;

import java.util.Date;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.NotUniqueUserException;
import org.genesys.blocks.security.UserException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.model.BasicUser.AccountType;
import org.genesys.blocks.security.persistence.AclEntryPersistence;
import org.genesys.blocks.security.service.BasicUserService;
import org.genesys.blocks.security.service.PasswordPolicy;
import org.genesys.blocks.security.service.PasswordPolicy.PasswordPolicyException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.NoOpPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class BasicUserServiceImpl.
 *
 * @param <R> the generic type
 * @param <T> the generic type
 */
@Transactional(readOnly = true)
public abstract class BasicUserServiceImpl<R extends GrantedAuthority, T extends BasicUser<R>> implements BasicUserService<R, T> {

	/** The Constant LOG. */
	public static final Logger LOG = LoggerFactory.getLogger(BasicUserServiceImpl.class);

	/** The Constant THIS_IS_NOT_A_PASSWORD. */
	/// A non-password used for system and Google accounts
	private static final String THIS_IS_NOT_A_PASSWORD = "THIS-IS-NOT-A-PASSWORD";

	/** The account lockout time. */
	private long accountLockoutTime = 5 * 60 * 1000;

	/** The user repository. */
	@Autowired
	private JpaRepository<T, Long> userRepository;

	/** The password encoder. */
	@Autowired(required = false)
	protected final PasswordEncoder passwordEncoder = NoOpPasswordEncoder.getInstance();

	/** The password policy. */
	@Autowired(required = false)
	private PasswordPolicy passwordPolicy;

	@Autowired(required = false)
	protected AclEntryPersistence aclEntryRepository;

	/**
	 * Sets the account lockout time.
	 *
	 * @param accountLockoutTime the new account lockout time
	 */
	public void setAccountLockoutTime(final long accountLockoutTime) {
		this.accountLockoutTime = accountLockoutTime;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#getDefaultUserRole()
	 */
	@Override
	public abstract R getDefaultUserRole();

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#listAvailableRoles()
	 */
	@Override
	public abstract List<R> listAvailableRoles();

	/*
	 * (non-Javadoc)
	 * @see org.springframework.security.core.userdetails.UserDetailsService#
	 * loadUserByUsername(java.lang.String)
	 */
	@Override
	public UserDetails loadUserByUsername(final String username) throws UsernameNotFoundException {
		T user = getUserByEmail(username);
		if (user == null) {
			throw new UsernameNotFoundException(username);
		}
		return user;
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.BasicUserService#getUser(long)
	 */
	@Override
	public T getUser(final long id) {
		final T user = userRepository.findOne(id);
		return deepLoad(user);
	}

	public T deepLoad(final T user) {
		if (user != null) {
			user.getRoles().size();
		}
		return user;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#updateUser(org.genesys.
	 * blocks.security.model.BasicUser, java.lang.String, java.lang.String)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') || principal.id == #user.id")
	public T updateUser(T user, final String email, final String fullName) throws NotUniqueUserException, UserException {
		// reload
		user = userRepository.findOne(user.getId());

		if (!StringUtils.equals(email, user.getEmail()) && getUserByEmail(email) != null) {
			throw new NotUniqueUserException("Email address already registered");
		}

		user.setEmail(email);
		user.setFullName(fullName);

		return deepLoad(userRepository.save(user));
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#deleteUser(org.genesys.
	 * blocks.security.model.BasicUser)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public void deleteUser(final T user) {
		userRepository.delete(user);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#setRoles(org.genesys.
	 * blocks.security.model.BasicUser, java.util.Set)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public T setRoles(T user, final Set<R> newRoles) {
		user = userRepository.findOne(user.getId());

		// If roles match, do nothing
		if (newRoles.containsAll(user.getRoles()) && user.getRoles().containsAll(newRoles)) {
			LOG.debug("Roles {} match {}. No change.", newRoles, user.getRoles());
			return user;
		}

		user.getRoles().clear();
		user.getRoles().addAll(newRoles);
		user.getRoles().add(getDefaultUserRole());
		LOG.info("Setting roles for user {} to {}", user.getEmail(), user.getRoles());
		return deepLoad(userRepository.save(user));
	}

	/*
	 * (non-Javadoc)
	 * @see org.genesys.blocks.security.service.BasicUserService#changePassword(org.
	 * genesys.blocks.security.model.BasicUser, java.lang.String)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR') || principal.id == #user.id")
	public T changePassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			setPassword(user, password);
			return deepLoad(userRepository.save(user));
		} else {
			throw new PasswordPolicyException("Password can be set only for LOCAL account types");
		}
	}

	/**
	 * Sets the password.
	 *
	 * @param user the user
	 * @param password the password
	 * @throws PasswordPolicyException the password policy exception
	 */
	protected void setPassword(final T user, final String password) throws PasswordPolicyException {
		if (user.getAccountType() == AccountType.LOCAL) {
			assureGoodPassword(password);
			user.setPassword(password == null ? null : passwordEncoder.encode(password));
		} else {
			user.setPassword(THIS_IS_NOT_A_PASSWORD);
		}
	}

	/**
	 * Test if password passes the password policy (if set).
	 *
	 * @param password candidate password
	 * @throws PasswordPolicyException if password does not match policy
	 */
	public void assureGoodPassword(final String password) throws PasswordPolicyException {
		if (passwordPolicy != null) {
			passwordPolicy.assureGoodPassword(password);
		}
	}

	/**
	 * For internal use only.
	 *
	 * @param userId the user id
	 * @param locked the locked
	 * @throws NoUserFoundException the no user found exception
	 */
	@Override
	@Transactional
	public void setAccountLockLocal(final long userId, final boolean locked) throws NoUserFoundException {
		final T user = getUser(userId);
		if (locked) {
			// Lock for account until some time
			user.setLockedUntil(new Date(System.currentTimeMillis() + accountLockoutTime));
			LOG.warn("Locking user account for user=" + user.getEmail() + "  until=" + user.getLockedUntil());
		} else {
			LOG.warn("Unlocking user account for user=" + user.getEmail());
			user.setLockedUntil(null);
		}
		userRepository.save(user);
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.genesys.blocks.security.service.BasicUserService#setAccountLock(long,
	 * boolean)
	 */
	@Override
	@Transactional
	@PreAuthorize("hasRole('ADMINISTRATOR')")
	public void setAccountLock(final long userId, final boolean locked) throws NoUserFoundException {
		setAccountLockLocal(userId, locked);
	}

	@Override
	@Transactional
	public T setAccountType(T user, AccountType accountType) {
		T u = userRepository.findOne(user.getId());
		u.setAccountType(accountType);

		if (accountType != AccountType.LOCAL) {
			user.setPassword(THIS_IS_NOT_A_PASSWORD);
		}

		return userRepository.save(u);
	}
}
