/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service;

import java.util.List;
import java.util.Map;

import org.genesys.blocks.security.model.AclAwareModel;
import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclObjectIdentity;
import org.genesys.blocks.security.model.AclSid;
import org.springframework.security.acls.model.Permission;

/**
 * The Interface CustomAclService.
 */
public interface CustomAclService {

	/**
	 * Get SID by ID
	 * 
	 * @param id -- ID of an AclSid entity
	 * @return persisted AclSid entity with specified id
	 */
	AclSid getSid(Long id);
	
	/**
	 * Gets the sid of the specified authority
	 *
	 * @param authority the authority
	 * @return the authority sid
	 */
	AclSid getAuthoritySid(String authority);

	/**
	 * Adds the creator permissions.
	 *
	 * @param target the target
	 */
	void addCreatorPermissions(AclAwareModel target);

	/**
	 * Removes the permissions on ACL model.
	 *
	 * @param target the target
	 */
	void removePermissions(AclAwareModel target);

	/**
	 * Removes the all permissions of SID.
	 *
	 * @param sid the sid
	 */
	void removePermissionsFor(AclSid sid);

	/**
	 * Gets the object identity.
	 *
	 * @param clazz the clazz
	 * @param id the id
	 * @return the object identity
	 */
	AclObjectIdentity getObjectIdentity(String clazz, long id);

	/**
	 * Gets the object identity.
	 *
	 * @param entity the entity
	 * @return the object identity
	 */
	AclObjectIdentity getObjectIdentity(AclAwareModel entity);

	/**
	 * Gets the available permissions.
	 *
	 * @param className the class name
	 * @return the available permissions
	 */
	Permission[] getAvailablePermissions(String className);

	/**
	 * Gets the permissions.
	 *
	 * @param id the id
	 * @param className the class name
	 * @return the permissions
	 */
	Map<String, Map<Integer, Boolean>> getPermissions(long id, String className);

	/**
	 * Gets the permissions.
	 *
	 * @param entity the entity
	 * @return the permissions
	 */
	Map<String, Map<Integer, Boolean>> getPermissions(AclAwareModel entity);

	/**
	 * Gets the acl entries.
	 *
	 * @param objectIdentity the object identity
	 * @return the acl entries
	 */
	List<AclEntry> getAclEntries(AclObjectIdentity objectIdentity);

	/**
	 * Update permissions.
	 *
	 * @param entity the entity
	 * @param sid the sid
	 * @param permissionMap the permission map
	 */
	void updatePermissions(AclAwareModel entity, AclSid sid, Map<Integer, Boolean> permissionMap);

	void updatePermissions(AclObjectIdentity objectIdentity, AclSid sid, Map<Integer, Boolean> permissionMap);

	/**
	 * Gets the acl entries.
	 *
	 * @param entity the entity
	 * @return the acl entries
	 */
	List<AclEntry> getAclEntries(AclAwareModel entity);

	/**
	 * Gets the sids.
	 *
	 * @param id the id
	 * @param className the class name
	 * @return the sids
	 */
	List<AclSid> getSids(long id, String className);

	/**
	 * Gets the sids.
	 *
	 * @param entity the entity
	 * @return the sids
	 */
	List<AclSid> getSids(AclAwareModel entity);

	/**
	 * Adds the permissions.
	 *
	 * @param objectIdIdentity the object id identity
	 * @param className the class name
	 * @param sid TODO
	 * @param permissions the permissions
	 * @return true, if successful
	 */
	boolean addPermissions(long objectIdIdentity, String className, AclSid sid, Map<Integer, Boolean> permissions);

	/**
	 * Adds the permissions.
	 *
	 * @param entity the entity
	 * @param sid the sid
	 * @param permissionMap the permission map
	 * @return true, if successful
	 */
	boolean addPermissions(AclAwareModel entity, AclSid sid, Map<Integer, Boolean> permissionMap);

	/**
	 * Ensure object identity.
	 *
	 * @param className the class name
	 * @param objectIdIdentity the object id identity
	 * @return the acl object identity
	 */
	AclObjectIdentity ensureObjectIdentity(String className, long objectIdIdentity);

	/**
	 * List IDs of the specified class for the SID with specified permissions.
	 *
	 * @param clazz the clazz
	 * @param sid the sid
	 * @param permission the permission
	 * @return the list
	 */
	List<Long> listObjectIdentityIdsForSid(Class<? extends AclAwareModel> clazz, AclSid sid, Permission permission);

	/**
	 * Utility method that creates a non-granting permission map
	 * 
	 * @return Map with all available permissions, set to false
	 */
	Map<Integer, Boolean> blankPermissionsMap();

}
