/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.service;

import java.util.List;
import java.util.Set;

import org.genesys.blocks.security.NoUserFoundException;
import org.genesys.blocks.security.NotUniqueUserException;
import org.genesys.blocks.security.UserException;
import org.genesys.blocks.security.model.BasicUser;
import org.genesys.blocks.security.model.BasicUser.AccountType;
import org.genesys.blocks.security.service.PasswordPolicy.PasswordPolicyException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetailsService;

/**
 * The Interface BasicUserService.
 *
 * @param <R> the generic type
 * @param <T> the generic type
 */
public interface BasicUserService<R extends GrantedAuthority, T extends BasicUser<R>> extends UserDetailsService {

	/**
	 * Get User by id.
	 *
	 * @param id the id
	 * @return the user or <code>null</code>
	 */
	T getUser(long id);

	/**
	 * Get User by email.
	 *
	 * @param email the email
	 * @return the user
	 */
	T getUserByEmail(String email);

	/**
	 * Create a new user account with {@link #getDefaultUserRole()} assigned.
	 *
	 * @param email unique email address
	 * @param fullName Full name
	 * @param password initial account password
	 * @param accountType TODO
	 * @return the new user
	 * @throws NotUniqueUserException user is not unique in the system
	 * @throws PasswordPolicyException password violates current password policy
	 * @throws UserException any other exception
	 */
	T createUser(String email, String fullName, String password, BasicUser.AccountType accountType) throws NotUniqueUserException, PasswordPolicyException, UserException;

	/**
	 * Grant specified roles to user. The {@link #getDefaultUserRole()} will be
	 * added if missing.
	 *
	 * @param user the user
	 * @param roles the roles
	 * @return the updated user
	 */
	T setRoles(T user, Set<R> roles);

	/**
	 * Update user information.
	 *
	 * @param user the user
	 * @param email new email address
	 * @param fullName new fullName
	 * @return the updated user
	 * @throws NotUniqueUserException when email address is already registered 
	 * @throws UserException a generic user service exception
	 */
	T updateUser(T user, String email, String fullName) throws NotUniqueUserException, UserException;

	/**
	 * Change password.
	 *
	 * @param user the user
	 * @param password new password
	 * @return the updated user
	 * @throws PasswordPolicyException the password policy exception
	 */
	T changePassword(T user, String password) throws PasswordPolicyException;

	/**
	 * Try to delete user.
	 *
	 * @param user user to delete
	 */
	void deleteUser(T user);

	/**
	 * Lock user account.
	 *
	 * @param userId the user id
	 * @param locked Is account locked
	 * @throws NoUserFoundException the no user found exception
	 */
	void setAccountLockLocal(long userId, boolean locked) throws NoUserFoundException;

	/**
	 * Sets the account lock.
	 *
	 * @param userId the user id
	 * @param locked the locked
	 * @throws NoUserFoundException the no user found exception
	 */
	void setAccountLock(long userId, boolean locked) throws NoUserFoundException;

	/**
	 * List available roles.
	 *
	 * @return the list
	 */
	List<R> listAvailableRoles();

	/**
	 * Gets the default user role.
	 *
	 * @return the default user role
	 */
	R getDefaultUserRole();

	/**
	 * Sets the account type.
	 *
	 * @param user the user
	 * @param accountType the account type
	 * @return the t
	 */
	T setAccountType(T user, AccountType accountType);
}
