/*
 * Copyright 2017 Global Crop Diversity Trust
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.genesys.blocks.security.persistence;

import java.util.List;

import org.genesys.blocks.security.model.AclEntry;
import org.genesys.blocks.security.model.AclObjectIdentity;
import org.genesys.blocks.security.model.AclSid;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;

/**
 * The Interface AclEntryPersistence.
 */
public interface AclEntryPersistence extends JpaRepository<AclEntry, Long> {

	/**
	 * Find by object identity.
	 *
	 * @param aclObjectIdentity the acl object identity
	 * @return the list
	 */
	@Query("select ae from AclEntry ae where ae.aclObjectIdentity = :aclObjectIdentity")
	List<AclEntry> findByObjectIdentity(@Param("aclObjectIdentity") AclObjectIdentity aclObjectIdentity);

	/**
	 * List by SID and Object Identity
	 * 
	 * @param sid - SID
	 * @param objectIdentity the ACL object identity
	 * @return list of ACL entries for specified SID and OID
	 */
	@Query("select ae from AclEntry ae where ae.aclSid = :aclSid and ae.aclObjectIdentity = :aclObjectIdentity")
	List<AclEntry> findBySidAndObjectIdentity(@Param("aclSid") AclSid sid, @Param("aclObjectIdentity") AclObjectIdentity objectIdentity);

	/**
	 * 
	 * /** Find IDs of object of aclClass for sid with specified permissions.
	 *
	 * @param sid - SID
	 * @param aclClass - class of domain object
	 * @param mask - permissions
	 * @return - returns lists of user's permissions on domain class
	 */
	@Query("select ae.aclObjectIdentity.objectIdIdentity from AclEntry ae where ae.aclSid=?1 and ae.aclObjectIdentity.aclClass.aclClass=?2 and ae.mask=?3 and ae.granting=true")
	List<Long> findObjectIdentitiesForSidAndAclClassAndMask(AclSid sid, String aclClass, long mask);

	/**
	 * Calculates max. ace_order for acl_object_identity to avoid DuplicateIndex
	 * exception (acl_object_identity + ace_order is unique index)
	 *
	 * @param aclObjectEntityId - id of acl_object_identity table
	 * @return - max. ace_order value for current objectIdentityId
	 */
	@Query("select max(ae.aceOrder) from AclEntry ae join ae.aclObjectIdentity aoi where aoi.id = ?1")
	Long getMaxAceOrderForObjectEntity(long aclObjectEntityId);

	/**
	 * Gets the sids.
	 *
	 * @param objectIdentity the object identity
	 * @return the sids
	 */
	@Query("select distinct ae.aclSid from AclEntry ae where ae.aclObjectIdentity = :objectIdentity")
	List<AclSid> getSids(@Param("objectIdentity") AclObjectIdentity objectIdentity);

	/**
	 * Delete AclEntries for a SID
	 * 
	 * @param <T> type extending AclSid (User, OAuthClient)
	 * @param sid the SID
	 * @return number of deleted entries
	 */
	@Modifying
	@Query("delete from AclEntry ae where ae.aclSid = :sid")
	<T extends AclSid> int deleteForSid(@Param("sid") T sid);

}
